package com.twitch.ipdev.flexo.pipelines.deploy

import groovy.json.JsonOutput
import org.apache.commons.codec.digest.DigestUtils

class DeployECS implements Serializable {

  // Expects env.ecrImage to exist as the image that is being deployed.
  def static Closure deploy(String ecsAssumeRoleArn, String ecsTaskFamily, String codeDeployApplication, String codeDeployDeploymentGroup) {
    return {
      // create new task definition
      // first get temporary credentials via ecsAssumeRoleArn
      def stsJsonOutput = sh(script: "#!/bin/sh -e\naws sts assume-role --role-arn $ecsAssumeRoleArn --role-session-name flexo-ecs-task-update", returnStdout: true)
      def tempCredentials = readJSON(text: stsJsonOutput)

      // // then use those credentials
      env.AWS_ACCESS_KEY_ID = tempCredentials.Credentials.AccessKeyId
      env.AWS_SECRET_ACCESS_KEY = tempCredentials.Credentials.SecretAccessKey
      env.AWS_SESSION_TOKEN = tempCredentials.Credentials.SessionToken

      // debug statement to show that we've assumed role correctly
      sh 'aws sts get-caller-identity'

      // read current task definition via aws cli
      def taskListJsonOutput = sh(script: "aws ecs list-task-definitions --family-prefix $ecsTaskFamily --sort DESC --max-items 1", returnStdout: true)
      def taskList = readJSON(text: taskListJsonOutput)
      def previousTaskDefinitonArn = taskList.taskDefinitionArns[0]

      def taskDefinitionJsonOutput = sh(script: "aws ecs describe-task-definition --task-definition $previousTaskDefinitonArn", returnStdout: true)
      def taskDefinition = readJSON(text: taskDefinitionJsonOutput)

      // update with the latest ECR image from this build, clean up unneeded fields
      taskDefinition.taskDefinition.taskDefinitionArn = null
      taskDefinition.taskDefinition.containerDefinitions[0].image = env.ecrImage
      taskDefinition.taskDefinition.revision = null
      taskDefinition.taskDefinition.status = null
      taskDefinition.taskDefinition.requiresAttributes = null
      taskDefinition.taskDefinition.compatibilities = null
      writeJSON(file: 'taskDefinition.json', json: taskDefinition.taskDefinition)

      // register new task definition
      def registerTaskDefinitionJsonOut = sh(script: "aws ecs register-task-definition --cli-input-json file://taskDefinition.json", returnStdout: true)
      def newTaskDefinition = readJSON(text: registerTaskDefinitionJsonOut)
      def newTaskDefinitionArn = newTaskDefinition.taskDefinition.taskDefinitionArn

      // update appspec.yml and create deployment
      def appspec = readYaml(file: 'appspec.yml')
      appspec.Resources[0].TargetService.Properties.TaskDefinition = newTaskDefinitionArn
      writeYaml(file: 'appspec.yml', data: appspec, overwrite: true)

      def appspecJSON = JsonOutput.toJson(appspec)
      def appspecJSONSHA256 = org.apache.commons.codec.digest.DigestUtils.sha256Hex(appspecJSON)

      def createDeploymentJsonSkeleton = sh(script: "aws deploy create-deployment --generate-cli-skeleton", returnStdout: true)
      def createDeployment = readJSON(text: createDeploymentJsonSkeleton, returnPojo: true)
      createDeployment.applicationName = codeDeployApplication
      createDeployment.deploymentGroupName = codeDeployDeploymentGroup
      createDeployment.revision.revisionType = 'AppSpecContent'
      createDeployment.revision.appSpecContent.sha256 = appspecJSONSHA256
      createDeployment.revision.appSpecContent.content = appspecJSON

      // Remove keys we don't need
      ['s3Location', 'gitHubLocation', 'string'].each {
        createDeployment.revision.remove(it)
      }
      ['targetInstances', 'autoRollbackConfiguration', 'updateOutdatedInstancesOnly', 'fileExistsBehavior',
        'description', 'ignoreApplicationStopFailures', 'deploymentConfigName'].each {
          createDeployment.remove(it)
      }

      writeJSON(file: 'createDeployment.json', json: createDeployment)

      // Finally... trigger the deployment
      sh "aws deploy create-deployment --cli-input-json file://createDeployment.json"
      env.AWS_ACCESS_KEY_ID = ""
      env.AWS_SECRET_ACCESS_KEY = ""
      env.AWS_SESSION_TOKEN = ""
    }
  }
}