import * as asg from '@aws-cdk/aws-autoscaling';
import * as cloudwatch from '@aws-cdk/aws-cloudwatch';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as ecs from '@aws-cdk/aws-ecs';
import * as iam from '@aws-cdk/aws-iam';
import * as log from '@aws-cdk/aws-logs';
import * as cdk from '@aws-cdk/core';
import * as easymode from '@twitch/easymode';
import * as jenkinsConfig from '../jenkins-config/jenkins-config';

/**
 * JenkinsEcsProps extends AutoScallingGroupProps but omits userData in favor of addCommandsToUserData();
 */
export interface JenkinsSlavesEcsProps extends Omit<asg.AutoScalingGroupProps, "userData" >,
    Omit<log.LogGroupProps, "logGroupName">, Omit<cloudwatch.DashboardProps, "dashboardName"> {
  /**
   * String: Denoting which jenkins environment you're connecting to.
   * @default: production
   */
  readonly jenkinsEnvironment?: "production" | "staging" ;

  /**
   * The name of the Flexo ECS Cluster
   * @default: flexo-ECS-<region>-<account number>
   */
  readonly ecsClusterName?: string;

  /**
   * The log group you want the ECS cluster, and jobs to log to
   */
  readonly logGroupName: string;

  /**
   * The dashboard name that will be created in CloudWatch
   */
  readonly dashboardName: string;
}

/**
 * Create an ECS cluster in a tiny bubble
 * This will setup an ASG, and Private Link
 * to the Jenkins master for the specified environment
 * staging, or production
 */
export class JenkinsSlavesEcs extends cdk.Construct {

  public readonly asg: asg.AutoScalingGroup;
  public readonly ecsCluster: ecs.Cluster;
  public readonly vpc: ec2.IVpc;
  public readonly jenkinsMasterEndpoint: easymode.PrivateLinkEndpoint;

  protected readonly _JENKINS_ENV: string;
  protected readonly _FLEXO_ECS_CLUSTERNAME: string;

  constructor(scope: cdk.Construct, id: string, props: JenkinsSlavesEcsProps) {
    super(scope, id);

    // setup urls to download the ssm and cloudwatch agents respectively
    const SSM_AGENT_RPM = 'https://s3.amazonaws.com/ec2-downloads-windows/SSMAgent/latest/linux_amd64/amazon-ssm-agent.rpm';
    const CLOUDWATCH_AGENT_RPM = 'https://s3.amazonaws.com/amazoncloudwatch-agent/amazon_linux/amd64/latest/amazon-cloudwatch-agent.rpm';

    this._JENKINS_ENV = props.jenkinsEnvironment ? props.jenkinsEnvironment : "production";
    this._FLEXO_ECS_CLUSTERNAME = props.ecsClusterName ?
      props.ecsClusterName : `flexo-ECS-${cdk.Stack.of(this).region}-${cdk.Stack.of(this).account}`;

    // Cloudwatch instance configuration
    const cloudwatchconfig = {
      metrics: {
        namespace: 'JenkinsECSAgent',
        aggregation_dimensions: [
          ['App', 'Environment']
        ],
        metrics_collected: {
          mem: {
            measurement: [
              'used_percent'
            ],
            append_dimensions: {
              App: 'JenkinsECS',
              Environment: this._FLEXO_ECS_CLUSTERNAME
            },
          },
          swap: {
            measurement: [
              'used_percent'
            ],
            append_dimensions: {
              App: 'JenkinsECS',
              Environment: this._FLEXO_ECS_CLUSTERNAME
            },
          },
          disk: {
            resources: [
              '/'
            ],
            measurement: [
              'free', 'total', 'used', 'used_percent'
            ],
            append_dimensions: {
              App: 'JenkinsECS',
              Environment: this._FLEXO_ECS_CLUSTERNAME
            },
          }
        }
      }
    };

    // Create a role for Jenkins to assume
    new iam.Role(this, 'JenkinsAssumeRole', {
      roleName: `JenkinsMaster${this._JENKINS_ENV}`,
      assumedBy: new iam.CompositePrincipal(new iam.ArnPrincipal(jenkinsConfig._JENKINS_CONFIG[this._JENKINS_ENV].assumeRole)),
      path: '/',
      inlinePolicies: {
        JenkinsAssumeRolePolicy: new iam.PolicyDocument({
          statements: [
            new iam.PolicyStatement({
              effect: iam.Effect.ALLOW,
              actions: [
                    "ecs:RegisterTaskDefinition",
                    "ecs:ListClusters",
                    "ecs:DescribeContainerInstances",
                    "ecs:ListTaskDefinitions",
                    "ecs:DescribeTaskDefinition",
                    "ecs:DeregisterTaskDefinition",
                    "ecs:ListContainerInstances",
                    "ecs:RunTask",
                    "ecs:StartTask",
                    "ecs:StopTask",
                    "ecs:DescribeTasks"
              ],
              // setting the resource to anything other than '*'
              // This action does not support resource-level permissions.
              // Policies granting access must specify "*" in the resource element.
              resources: ["*"]
            }),
          ]
        })
      }
    });

    // Create the auto scale group
    this.asg = new asg.AutoScalingGroup(this, 'JenkinsWorkersASG', props);

    // setup userdata for the instances
    this.addCommandsToUserData(
        "yum -y update",
        // Install the SSM agent
        `yum install -y ${SSM_AGENT_RPM}`,

        // Make sure the SSM agent is installed and enabled
        "systemctl restart amazon-ssm-agent",
        "systemctl enable amazon-ssm-agent",

        // install cloudwatch agent
        `yum install ${CLOUDWATCH_AGENT_RPM}`,

        // create the cloudwatch configuration file
        "mkdir /opt/aws/amazon-cloudwatch-agent/etc",
        `echo ${JSON.stringify(cloudwatchconfig)} >/opt/aws/amazon-cloudwatch-agent/etc/amazon-cloudwatch-agent.json`,

        // make sure cloudwatch starts on boot
        "systemctl enable amazon-cloudwatch-agent",
        "systemctl start amazon-cloudwatch-agent"
    );

    // create a policy so that the instances can
    // send logs to cloudwatch
    const logPolicy = new iam.PolicyStatement();
    logPolicy.addActions(
        "logs:CreateLogGroup",
        "logs:CreateLogStream",
        "logs:PutLogEvents",
        "logs:DescribeLogStreams"
    );
    logPolicy.addResources("arn:aws:logs:*:*:*");

    // add the log policy to the asg
    this.asg.addToRolePolicy(logPolicy);

    // Adding these managed policies for using SSM
    // using the following syntax `aws ssm start-session --target <instance>`
    this.asg.role.addManagedPolicy({managedPolicyArn: 'arn:aws:iam::aws:policy/AmazonSSMManagedInstanceCore'});
    this.asg.role.addManagedPolicy({managedPolicyArn: 'arn:aws:iam::aws:policy/CloudWatchAgentServerPolicy'});

    // create the ECS cluster
    this.ecsCluster = new ecs.Cluster(this, 'JenkinsCluster', {
      clusterName: this._FLEXO_ECS_CLUSTERNAME
    });

    // create a security group to allow the jenkins master
    // and the agent instances to talk over the private link
    // connection
    const sg = new ec2.SecurityGroup(this, 'SG', {
      allowAllOutbound: true,
      description: 'SG to allow Jenkins Ports',
      securityGroupName: 'AllowJenkinsPorts',
      vpc: props.vpc,
    });

    // Allow egress on both 56862 and 443
    sg.addEgressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(56862), 'IngressJNLP', false);
    sg.addEgressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(443), 'IngressHTTPS', false);

    // Add the security group to the ASG
    this.asg.addSecurityGroup(sg);

    // Add the ASG to the Cluster
    this.ecsCluster.addAutoScalingGroup(this.asg);

    // Create the private link endpoint so the the agents can communicate with the Jenkins master in the
    // specified environment
    this.jenkinsMasterEndpoint = new easymode.PrivateLinkEndpoint(this, 'JenkinsMasterEndpoint', {
        vpc: props.vpc,
        service: {
          name: jenkinsConfig._JENKINS_CONFIG[this._JENKINS_ENV].vpcEndpoint,
          port: 443
        },
        // configure a private hosted zone. This allows the jenkins/jnlp-slave
        // containers to connect to a DNS name, for example:
        // https://staging.jenkins.xarth.tv
        hostedZones: [{
          zoneName: this._JENKINS_ENV === "production" ?  'jenkins.xarth.tv' : `${this._JENKINS_ENV}.jenkins.xarth.tv`
        }]
      });

    // Allow 443 and 56862
    this.jenkinsMasterEndpoint.connections.allowFrom(this.asg, new ec2.Port({
      protocol: ec2.Protocol.TCP,
      toPort: 443,
      fromPort: 443,
      stringRepresentation: 'Jenkins443'
    }));
    this.jenkinsMasterEndpoint.connections.allowFrom(this.asg, new ec2.Port({
      protocol: ec2.Protocol.TCP,
      toPort: 56862,
      fromPort: 56862,
      stringRepresentation: 'JenkinsJNLP'
    }));

    /**
     * Create a log group in cloudwatch
     */
    new log.LogGroup(this, 'JenkinsECSLogGroup', props);

    // create a dashboard so that the performance of the ECS instances can be monitored
    const dashboard = new cloudwatch.Dashboard(this, 'Jenkins-ECS-Agents', props);

    // Create dashboard widgets for each metric(s) to go on the dashboard
    dashboard.addWidgets(new cloudwatch.GraphWidget({
      title: "Flexo Jenkins ECS Agents - EC2 CPU Utilization",
      left: [new cloudwatch.Metric({
        namespace: 'AWS/EC2',
        metricName: 'CPUUtilization',
        dimensions: {AutoScalingGroupName: `${this.asg.autoScalingGroupName}`},
      })],
      height: 6,
      width: 12
    }));

    dashboard.addWidgets(new cloudwatch.GraphWidget({
      title: "Flexo Jenkins Agents - EC2 Network Utilization",
      left: [new cloudwatch.Metric({
        namespace: 'AWS/EC2',
        metricName: 'NetworkIn',
        dimensions: {AutoScalingGroupName: `${this.asg.autoScalingGroupName}`}
      }), new cloudwatch.Metric({
        namespace: 'AWS/EC2',
        metricName: 'NetworkOut',
        dimensions: {AutoScalingGroupName: `${this.asg.autoScalingGroupName}`}
      })],
      height: 6,
      width: 12
    }));

    dashboard.addWidgets(new cloudwatch.GraphWidget({
      title: "Flexo Jenkins Agents - EC2 Disk Utilization",
      left: [new cloudwatch.Metric({
        namespace: 'AWS/EC2',
        metricName: 'DiskWriteBytes',
        dimensions: {AutoScalingGroupName: `${this.asg.autoScalingGroupName}`}
      }), new cloudwatch.Metric({
        namespace: 'AWS/EC2',
        metricName: 'DiskReadBytes',
        dimensions: {AutoScalingGroupName: `${this.asg.autoScalingGroupName}`}
      })],
      height: 6,
      width: 12
    }));

    dashboard.addWidgets(new cloudwatch.GraphWidget({
      title: "Flexo Jenkins Agents - ECS Memory Utilization",
      left: [new cloudwatch.Metric({
        namespace: 'AWS/ECS',
        metricName: 'MemoryReservation',
        dimensions: {ClusterName: `${this.ecsCluster.clusterName}`}
      }), new cloudwatch.Metric({
        namespace: 'AWS/ECS',
        metricName: 'MemoryUtilization',
        dimensions: {ClusterName: `${this.ecsCluster.clusterName}`}
      })],
      height: 6,
      width: 12
    }));

    dashboard.addWidgets(new cloudwatch.GraphWidget({
      title: "Flexo Jenkins Agents - ECS CPU Utilization",
      left: [new cloudwatch.Metric({
        namespace: 'AWS/ECS',
        metricName: 'CPUReservation',
        dimensions: {ClusterName: `${this.ecsCluster.clusterName}`}
      }), new cloudwatch.Metric({
        namespace: 'AWS/ECS',
        metricName: 'CPUUtilization',
        dimensions: {ClusterName: `${this.ecsCluster.clusterName}`}
      })],
      height: 6,
      width: 12
    }));
  }

  // set various values for each jenkins environment
  public addCommandsToUserData(...commands: string[]): void {
    this.asg.userData.addCommands(...commands);
  }
}
