# @twitch/flexo/JenkinsSlavesSwarmASG

Automatically add a Jenkins Slave to the Flexo Jenkins master.  Your slave will be running in a tiny bubble and communicate with Jenkins via a PrivateLink.  This automation will also set up some very basic dashboards in cloudwatch.

We recommend using AL2.

### Before running the automation

Contact #developer-infra - we'll need to onboard your new tiny bubble account.

* Please give the POSIX group `twitch-devinfra` read access to your tiny bubble account in Isengard
* We'll provide you with a username/password that will need to be stored in your account's AWS Secret Manager.
* #developer-infra will follow this guide [here](https://wiki.twitch.com/display/PS/Onboarding+to+Flexo) and [here](https://wiki.twitch.com/display/PS/Onboarding+to+Flexo#OnboardingtoFlexo-OnboardinganASGTinyBubble).

To add this secret:

```bash
aws secretsmanager create-secret --name jenkins-slave-credentials
aws secretsmanager put-secret-value --secret-id jenkins-slave-credentials \
    --secret-string '{"jenkinsSlaveUser":"bob", "jenkinsSlavePassword":"xxxxxxxxxxxxx"}'
```

Once this secret is created - the ARN for this secret will be the value for `secretArn` in the construct's props.

Give #developer-infra a heads up when you'll be running your automation to spin up Jenkins slaves, and we'll ensure your PrivateLink request is accepted and approved.  Please also give us notice for what your `swarmLabels` will be (See [Guidance on creating your swarm](#guidance-on-creating-your-swarm))

### Guidance on creating your swarm

To differentiate your nodes in the Jenkins cluster, we'll need to make sure you have a unique `swarmLabel` for your nodes.  `swarmLabels` supports a whitespace-separated list of labels to be assigned to all the slaves.

We recommend setting your label to `swarmLabels: '[team]-[service]' [any other necessary unique tags]`

For example, if `Team Rocket` was creating a swarm for their `BlastingOff` Service, and wanted to note that they were spinning up Windows agents - 

```
// Will apply 2 labels to the swarm "TeamRocket-BlastingOff" and "Windows"
swarmLabels: 'TeamRocket-BlastingOff Windows'
```

### To create an AL2 Jenkins Swarm Slave (recommended)

```typescript
import * as cdk from '@aws-cdk/core';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as flexo from '@twitch/flexo';


export class SwarmCdkEasymodeStack extends cdk.Stack {
  constructor(scope: cdk.Construct, id: string, props: cdk.StackProps) {
    super(scope, id, props);
    
    const vpc = new ec2.Vpc(this, 'vpc');
    
    const swarm = new flexo.JenkinsSlavesSwarmASGAmazonLinux2(this, 'flexo', {
      vpc,
      instanceType: ec2.InstanceType.of(ec2.InstanceClass.T2, ec2.InstanceSize.MICRO),
      desiredCapacity: 2,
      maxCapacity: 2,
      minCapacity: 2,
      vpcSubnets: {
        subnetType: ec2.SubnetType.PRIVATE
      },
      swarmLabels: 'TeamXyzSwarmNodes AnotherLabel',
      swarmDescription: 'Nodes to support TeamXyz',
      swarmExecutors: 3,
      secretArn: 'arn:aws:secretsmanager:us-west-2:11111111:secret:jenkins-slave-credentials-XYZABC',
      dashboardName: 'AL2JenkinsDashboard'
    });
    
    swarm.addCommandsToUserData(
    	'yum install mycoolpackage -y',
    	'mycoolpackage do-al2-stuff'
    );
  }
}
```

### To create an Ubuntu Jenkins Swarm Slave

```typescript
import * as cdk from '@aws-cdk/core';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as flexo from '@twitch-easymode/flexo';

export class UbuntuSwarmCdkEasymodeStack extends cdk.Stack {
  constructor(scope: cdk.Construct, id: string, props: VpcStackProps) {
    super(scope, id, props);
    
    const vpc = new ec2.Vpc(this, 'vpc');
    
    const swarm = new flexo.JenkinsSlavesSwarmASGUbuntu(this, 'flexoUbuntu', {
      vpc,
      machineImage: new ec2.GenericLinuxImage({
        // Ubuntu Server 18.04 LTS (HVM), SSD Volume Type
        "us-west-2" : "ami-06d51e91cea0dac8d"
      }),
      instanceType: ec2.InstanceType.of(ec2.InstanceClass.T2, ec2.InstanceSize.MICRO),
      desiredCapacity: 2,
      maxCapacity: 2,
      minCapacity: 2,
      vpcSubnets: {
        subnetType: ec2.SubnetType.PRIVATE
      },
      swarmLabels: 'UbuntuSwarmForCoolPeople',
      swarmDescription: 'Ubuntu Swarm for team ABC',
      swarmExecutors: 3,
      secretArn: 'arn:aws:secretsmanager:us-west-2:11111111:secret:jenkins-slave-credentials-XYZABC',
      dashboardName: 'UbuntuJenkinsDashboard'
    });
    
    swarm.addCommandsToUserData(
    	'apt-get install mycoolpackage -y',
    	'mycoolpackage do-ubuntu-stuff'
    );
  }
}
```
