import * as autoscaling from '@aws-cdk/aws-autoscaling';
import * as cloudwatch from '@aws-cdk/aws-cloudwatch';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as iam from '@aws-cdk/aws-iam';
import * as secretsmanager from '@aws-cdk/aws-secretsmanager';
import * as cdk from '@aws-cdk/core';
import * as easymode from '@twitch/easymode';
import * as jenkinsConfig from '../jenkins-config/jenkins-config';

/**
 * Default values for URLs/versions
 */
const _DEFAULT_SSM_AGENT_RPM = 'https://s3.amazonaws.com/ec2-downloads-windows/SSMAgent/latest/linux_amd64/amazon-ssm-agent.rpm';
const _DEFAULT_SWARM_VERSION = '3.17';
const _DEFAULT_SWARM_CLIENT_JAR = `https://repo.jenkins-ci.org/releases/org/jenkins-ci/plugins/swarm-client/${_DEFAULT_SWARM_VERSION}/swarm-client-${_DEFAULT_SWARM_VERSION}.jar`;
const _DEFAULT_JAR_DIRECTORY = '/opt/twitch/jenkins/';
const _DEFAULT_FSROOT = '/var/lib/jenkins/';
const _GHE_HOSTNAME = 'git-aws.internal.justin.tv';
const _GHE_VPC_ENDPOINT = 'com.amazonaws.vpce.us-west-2.vpce-svc-02d9c29bb653310a1';

/**
 * Properties to configure the baseline for Jenkins Swarm Slave Nodes
 * NOTE: Full options info for teh jenkins swarm can be found here: https://plugins.jenkins.io/swarm
 * NOTE: autoscaling.AutoScalingGroupProps.userData is omitted in favor of using FlexoTinyBubbleJenkinsSwarmSlaveProps.addCommandsToUserData()
 * NOTE: The secret required by this interface is a secret containing your jenkins user/password.
 *   The secret should be stored in secrets manager with the format:
 *    { jenkinsSlaveUser: "myUser", jenkinsSlavePassword: "myPassword" }
 */
interface JenkinsSlavesSwarmASGBaseProps extends
    secretsmanager.SecretAttributes, Omit<autoscaling.AutoScalingGroupProps, "userData">, cloudwatch.DashboardProps {
    /**
     * Whitespace-separated list of labels to be assigned for the slaves. Multiple labels are allowed.
     */
    readonly swarmLabels: string;

    /**
     * Description to be put on the slaves
     */
    readonly swarmDescription?: string;

    /**
     * Number of executors per slave
     * @default 1
     */
    readonly swarmExecutors?: number;

    /**
     * The mode controlling how Jenkins allocates jobs to slaves.
     * Can be either 'normal' (utilize this slave as much as possible) or 'exclusive' (leave this machine for tied jobs only).
     * @default "normal"
     */
    readonly swarmMode?: "normal" | "exclusive";

    /**
     * Any other options to append to the end of the swarm-client.jar invocation
     * See all the options here https://plugins.jenkins.io/swarm
     */
    readonly swarmExtraOptions?: string;

    /**
     * Directory where Jenkins places files
     * @default `${_DEFAULT_FSROOT}`
     */
    readonly swarmFsRoot?: string;

    /**
     * Whether to use the config for connecting to staging/production Jenkins (unless you know what you're doing, use production)
     * @default 'production'
     */
    readonly jenkinsEnvironment?: "staging" | "production";

    /**
     * URL to retrieve the updated SSM RPM
     * @default `${_DEFAULT_SSM_AGENT_RPM}`
     */
    readonly ssm_agent_rpm?: string;

    /**
     * Version of the Jenkins Swarm Client to use
     * @default `${_DEFAULT_SWARM_VERSION}`
     */
    readonly swarm_version?: string;

    // tslint:disable:max-line-length
    /**
     * The URL to get the Swarm Client Jar file
     * @default `https://repo.jenkins-ci.org/releases/org/jenkins-ci/plugins/swarm-client/${_DEFAULT_SWARM_VERSION}/swarm-client-${_DEFAULT_SWARM_VERSION}.jar`
     */
    readonly swarm_client_jar?: string;
    // tslint:enable:max-line-length

    /**
     * Directory to download jar files to
     * @default `${_DEFAULT_JAR_DIRECTORY}`
     */
    readonly jar_directory?: string;
}

/**
 * Create a Jenkins slave in a tiny bubble.
 * This will set up an ASG and a PrivateLink to the Jenkins Master
 * The ASG will automatically connect to Jenkins master
 */
class JenkinsSlavesSwarmASGBase extends cdk.Construct {
    protected readonly asg: autoscaling.AutoScalingGroup;
    protected readonly jenkinsMasterPrivateLinkEndpoint: easymode.PrivateLinkEndpoint;
    protected readonly ghePrivateLinkEndpoint: easymode.PrivateLinkEndpoint;
    protected readonly swarmOptions: string;

    protected readonly _SSM_AGENT_RPM: string;
    protected readonly _SWARM_VERSION: string;
    protected readonly _SWARM_CLIENT_JAR: string;
    protected readonly _JAR_DIRECTORY: string;
    protected readonly _JENKINS_ENV: string;
    protected readonly _FSROOT: string;

    constructor(scope: cdk.Construct, id: string, props: JenkinsSlavesSwarmASGBaseProps) {
        super(scope, id);

        // Set up some values either from props or default values
        this._SSM_AGENT_RPM = props.ssm_agent_rpm ? props.ssm_agent_rpm : _DEFAULT_SSM_AGENT_RPM;
        this._SWARM_VERSION = props.swarm_version ? props.swarm_version : _DEFAULT_SWARM_VERSION;
        this._SWARM_CLIENT_JAR = props.swarm_client_jar ? props.swarm_client_jar : _DEFAULT_SWARM_CLIENT_JAR;
        this._JAR_DIRECTORY = props.jar_directory ? props.jar_directory : _DEFAULT_JAR_DIRECTORY;
        this._JENKINS_ENV = props.jenkinsEnvironment ? props.jenkinsEnvironment : "production";
        this._FSROOT = props.swarmFsRoot ? props.swarmFsRoot : _DEFAULT_FSROOT;

        // Set up options for the swarm jar
        this.swarmOptions = ` -fsroot ${this._FSROOT}`;
        this.swarmOptions = `${this.swarmOptions} -labels \\"${props.swarmLabels}\\"`;
        this.swarmOptions = props.swarmDescription  ? `${this.swarmOptions} -description \\"${props.swarmDescription}\\"` : `${this.swarmOptions}`;
        this.swarmOptions = props.swarmExecutors    ? `${this.swarmOptions} -executors ${props.swarmExecutors}`           : `${this.swarmOptions}`;
        this.swarmOptions = props.swarmMode         ? `${this.swarmOptions} -mode ${props.swarmMode}`                     : `${this.swarmOptions}`;
        this.swarmOptions = props.swarmExtraOptions ? `${this.swarmOptions} ${props.swarmExtraOptions}`                   : `${this.swarmOptions}`;

        // Jenkins Slave ASG
        this.asg = new autoscaling.AutoScalingGroup(this, 'JenkinsSlaveASG', props);

        // Jenkins Master PrivateLink
        this.jenkinsMasterPrivateLinkEndpoint = new easymode.PrivateLinkEndpoint(this, 'jenkinsMasterEndpoint', {
            vpc: props.vpc,
            service: {
                name: jenkinsConfig._JENKINS_CONFIG[this._JENKINS_ENV].vpcEndpoint,
                port: 443
            },
            hostedZones: [{
                zoneName: jenkinsConfig._JENKINS_CONFIG[this._JENKINS_ENV].hostname
            }]
        });

        // Allow ASG to reach PrivateLink
        this.jenkinsMasterPrivateLinkEndpoint.connections.allowFrom(this.asg, new ec2.Port({
            protocol: ec2.Protocol.TCP,
            toPort: 56862,
            fromPort: 56862,
            stringRepresentation: "JNLP port"
        }));

        // GHE PrivateLink
        this.ghePrivateLinkEndpoint = new easymode.PrivateLinkEndpoint(this, 'gheEndpoint', {
            vpc: props.vpc,
            service: {
                name: _GHE_VPC_ENDPOINT,
                port: 443
            },
            hostedZones: [{
                zoneName: _GHE_HOSTNAME
            }]
        });

        // Add SSM policy and Cloudwatch Policies by default
        this.asg.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        this.asg.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('CloudWatchAgentServerPolicy'));

        // Allow the ASG access to the secret jenkins credentials
        this.asg.role.addToPolicy(new iam.PolicyStatement({
            resources: [props.secretArn],
            actions: ["secretsmanager:GetSecretValue"]
        }));

        // Allow the agent to assume a role
        this.asg.role.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['sts:AssumeRole']
        }));

        // Create a dashboard so that the performance of the ECS instances can be monitored
        const dashboard = new cloudwatch.Dashboard(this, 'Jenkins-EC2-Agents-Dashboard', props);

        // Create dashboard widgets for each metric(s) to go on the dashboard
        dashboard.addWidgets(new cloudwatch.GraphWidget({
            title: "Flexo Jenkins Agents - EC2 CPU Utilization",
            left: [
                new cloudwatch.Metric({
                    namespace: 'AWS/EC2',
                    metricName: 'CPUUtilization',
                    dimensions: {AutoScalingGroupName: `${this.asg.autoScalingGroupName}`},
            })],
            height: 6,
            width: 12
        }));

        dashboard.addWidgets(new cloudwatch.GraphWidget({
            title: "Flexo Jenkins Agents - EC2 Network Utilization",
            left: [
                new cloudwatch.Metric({
                    namespace: 'AWS/EC2',
                    metricName: 'NetworkIn',
                    dimensions: {AutoScalingGroupName: `${this.asg.autoScalingGroupName}`}
                }),
                new cloudwatch.Metric({
                    namespace: 'AWS/EC2',
                    metricName: 'NetworkOut',
                    dimensions: {AutoScalingGroupName: `${this.asg.autoScalingGroupName}`}
            })],
            height: 6,
            width: 12
        }));

        dashboard.addWidgets(new cloudwatch.GraphWidget({
            title: "Flexo Jenkins Agents - EC2 Disk Utilization",
            left: [
                new cloudwatch.Metric({
                    namespace: 'AWS/EC2',
                    metricName: 'DiskWriteBytes',
                    dimensions: {AutoScalingGroupName: `${this.asg.autoScalingGroupName}`}
                }),
                new cloudwatch.Metric({
                    namespace: 'AWS/EC2',
                    metricName: 'DiskReadBytes',
                    dimensions: {AutoScalingGroupName: `${this.asg.autoScalingGroupName}`}
            })],
            height: 6,
            width: 12
        }));
    }

    public addCommandsToUserData(...commands: string[]): void {
        this.asg.userData.addCommands(...commands);
    }
}

/**
 * Options to bring up an AL2 Slave as a Jenkins Swarm Slave
 * @default machineImage is `new ec2.AmazonLinuxImage({ generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2 })`
 */
export interface JenkinsSlavesSwarmASGAmazonLinux2Props extends Omit<JenkinsSlavesSwarmASGBaseProps, "machineImage"> {
    /**
     * AMI to use.  MUST be AL2 based
     * @default new ec2.AmazonLinuxImage({ generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2 })
     */
    readonly machineImage?: ec2.IMachineImage;
}

/**
 * Create a Jenkins slave in a tiny bubble using Amazon Linux 2 as your hosts
 * This will set up an ASG and a PrivateLink to the Jenkins Master
 * The ASG will automatically connect to Jenkins master
 * To get just the userdata, you can access `FlexoTinyBubbleJenkinsSwarmSlaveAmazonLinux2.al2UserData`
 */
export class JenkinsSlavesSwarmASGAmazonLinux2  extends JenkinsSlavesSwarmASGBase {
    public readonly al2UserData: string[];

    constructor(scope: cdk.Construct, id: string, props: JenkinsSlavesSwarmASGAmazonLinux2Props) {
        super(scope, id, {
            machineImage: props.machineImage ?
                props.machineImage : new ec2.AmazonLinuxImage({ generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            ...props
        });

        // tslint:disable:max-line-length
        this.al2UserData = [
            'yum -y update',
            `yum install -y java-11-amazon-corretto-headless wget jq`,
            `yum install -y ${this._SSM_AGENT_RPM}`,
            'systemctl restart amazon-ssm-agent',
            'systemctl enable amazon-ssm-agent',

            'groupadd -g 61000 jenkins',
            'useradd -u 61000 -g jenkins -s /bin/bash -d /var/lib/jenkins jenkins',
            `mkdir -p ${this._JAR_DIRECTORY}`,
            `mkdir -p ${this._FSROOT }`,

            `wget ${this._SWARM_CLIENT_JAR} -O ${this._JAR_DIRECTORY}/swarm-client-${this._SWARM_VERSION}.jar`,
            `echo "JENKINSUSER=$(aws secretsmanager get-secret-value --secret-id ${props.secretArn} --region ${cdk.Stack.of(this).region} --query "SecretString" --output text | jq -r .jenkinsSlaveUser)" > ${this._JAR_DIRECTORY}/jenkinsEnv`,
            `echo "JENKINSPW=$(aws secretsmanager get-secret-value --secret-id ${props.secretArn} --region ${cdk.Stack.of(this).region} --query "SecretString" --output text | jq -r .jenkinsSlavePassword)" >> ${this._JAR_DIRECTORY}/jenkinsEnv`,
            `chown -R jenkins:jenkins ${this._JAR_DIRECTORY}`,
            `chown -R jenkins:jenkins ${this._FSROOT}`,

            `echo -e "
            [Unit]
            Description=Connect To Jenkins Master as a swarm slave

            [Service]
            User=jenkins
            EnvironmentFile=${this._JAR_DIRECTORY}/jenkinsEnv
            Type=simple
            ExecStart=/bin/bash -c \\
                '/usr/bin/java -jar ${this._JAR_DIRECTORY}/swarm-client-${this._SWARM_VERSION}.jar -retryBackOffStrategy exponential -master https://${jenkinsConfig._JENKINS_CONFIG[this._JENKINS_ENV].hostname}/ -username ` + '\\$JENKINSUSER' + ` -passwordEnvVariable JENKINSPW ${this.swarmOptions}'

            [Install]
            WantedBy=multi-user.target" > /etc/systemd/system/jenkins-slave.service`,
            `systemctl enable jenkins-slave`,
            `systemctl  start jenkins-slave`
        ];
        // tslint:enable:max-line-length
        this.addCommandsToUserData(...this.al2UserData);
    }
}

/**
 * Create a Jenkins slave in a tiny bubble using Ubuntu as your hosts
 * This will set up an ASG and a PrivateLink to the Jenkins Master
 * The ASG will automatically connect to Jenkins master
 * To get just the userdata, you can access `FlexoTinyBubbleJenkinsSwarmSlaveUbuntu.ubuntuUserData`
 * NOTE: FlexoTinyBubbleJenkinsSwarmSlaveBaseProps.machineImage MUST point to an Ubuntu AMI
 * NOTE: This has been texted as working on 'Ubuntu Server 18.04 LTS (ami-06d51e91cea0dac8d)'
 */
export class JenkinsSlavesSwarmASGUbuntu extends JenkinsSlavesSwarmASGBase {

    public readonly ubuntuUserData: string[];

    constructor(scope: cdk.Construct, id: string, props: JenkinsSlavesSwarmASGBaseProps) {
        super(scope, id, props);
        // tslint:disable:max-line-length
        this.ubuntuUserData = [
            'apt-get -y update',
            'snap install amazon-ssm-agent --classic',
            'snap install aws-cli --classic',
            'ln -s /snap/bin/aws /usr/bin/aws',
            'wget -O- https://apt.corretto.aws/corretto.key | apt-key add -',
            `add-apt-repository 'deb https://apt.corretto.aws stable main'`,
            'apt-get update',
            'apt-get install -y java-11-amazon-corretto-jdk wget jq',
            'systemctl start snap.amazon-ssm-agent.amazon-ssm-agent.service',
            'systemctl enable snap.amazon-ssm-agent.amazon-ssm-agent.service',

            'groupadd -g 61000 jenkins',
            'useradd -u 61000 -g jenkins -s /bin/bash -d /var/lib/jenkins jenkins',
            `mkdir -p ${this._JAR_DIRECTORY}`,
            `mkdir -p ${this._FSROOT }`,

            `wget ${this._SWARM_CLIENT_JAR} -O ${this._JAR_DIRECTORY}/swarm-client-${this._SWARM_VERSION}.jar`,
            `echo "JENKINSUSER=$(aws secretsmanager get-secret-value --secret-id ${props.secretArn} --region ${cdk.Stack.of(this).region} --query "SecretString" --output text | jq -r .jenkinsSlaveUser)" > ${this._JAR_DIRECTORY}/jenkinsEnv`,
            `echo "JENKINSPW=$(aws secretsmanager get-secret-value --secret-id ${props.secretArn} --region ${cdk.Stack.of(this).region} --query "SecretString" --output text | jq -r .jenkinsSlavePassword)" >> ${this._JAR_DIRECTORY}/jenkinsEnv`,
            `chown -R jenkins:jenkins ${this._JAR_DIRECTORY}`,
            `chown -R jenkins:jenkins ${this._FSROOT}`,

            `echo -e "
            [Unit]
            Description=Connect To Jenkins Master as a swarm slave

            [Service]
            User=jenkins
            EnvironmentFile=${this._JAR_DIRECTORY}/jenkinsEnv
            Type=simple
            ExecStart=/bin/bash -c \\
                '/usr/bin/java -jar ${this._JAR_DIRECTORY}/swarm-client-${this._SWARM_VERSION}.jar -retryBackOffStrategy exponential -master https://${jenkinsConfig._JENKINS_CONFIG[this._JENKINS_ENV].hostname}/ -username ` + '\\$JENKINSUSER' + ` -passwordEnvVariable JENKINSPW ${this.swarmOptions}'

            [Install]
            WantedBy=multi-user.target" > /etc/systemd/system/jenkins-slave.service`,
            `systemctl enable jenkins-slave`,
            `systemctl  start jenkins-slave`
        ];
        // tslint:enable:max-line-length
        this.addCommandsToUserData(...this.ubuntuUserData);
    }
}
