import * as cloudwatch from '@aws-cdk/aws-cloudwatch';
import * as actions from '@aws-cdk/aws-cloudwatch-actions';
import * as sns from '@aws-cdk/aws-sns';
import * as subs from '@aws-cdk/aws-sns-subscriptions';
import * as cdk from '@aws-cdk/core';

export interface TestMyAgentsProps extends cdk.StackProps {
  readonly pagerDutyErrorsIntegrationUrl: string,
  readonly cloudwatchNamespace?: string,
  readonly snsErrorTopic?: string,
  readonly dashboardProps?: cloudwatch.DashboardProps,
  readonly alarmThreshold?: number,
  readonly alarmEvaluationPeriods?: number,
  readonly alarmPeriodInMinutes?: number
}
export class TestMyAgents extends cdk.Construct {
  public readonly dashboard: cloudwatch.Dashboard;
  protected readonly cloudwatchNamespace: string;
  protected readonly snsErrorTopic: string;
  protected readonly alarmThreshold: number;
  protected readonly alarmEvaluationPeriods: number;
  protected readonly alarmPeriodInMinutes: number;
  protected readonly dashboardProps: cloudwatch.DashboardProps;
  protected readonly errorsTopic: sns.Topic;
  protected readonly errorsAction: actions.SnsAction;
  protected readonly cwAlarm: cloudwatch.Alarm;

  constructor(scope: cdk.Construct, id: string, props: TestMyAgentsProps) {
    super(scope, id);

    this.cloudwatchNamespace = props.cloudwatchNamespace ? props.cloudwatchNamespace : 'JenkinsAgents';
    this.snsErrorTopic = props.snsErrorTopic ? props.snsErrorTopic : 'JenkinsAgents';
    this.alarmThreshold = props.alarmThreshold ? props.alarmThreshold : 2;
    this.alarmEvaluationPeriods = props.alarmEvaluationPeriods ? props.alarmEvaluationPeriods : 2;
    this.alarmPeriodInMinutes = props.alarmPeriodInMinutes ? props.alarmPeriodInMinutes : 15;

    this.errorsTopic = new sns.Topic(this, this.snsErrorTopic);
    this.errorsAction = new actions.SnsAction(this.errorsTopic);
    this.errorsTopic.addSubscription(new subs.UrlSubscription(props.pagerDutyErrorsIntegrationUrl));

    this.cwAlarm =  new cloudwatch.Alarm(this, 'AgentFailures', {
        metric: new cloudwatch.Metric({
          namespace: this.cloudwatchNamespace,
          metricName: 'failures',
          statistic: cloudwatch.Statistic.SUM
        }),
        threshold: this.alarmThreshold,
        evaluationPeriods: this.alarmEvaluationPeriods,
        period: cdk.Duration.minutes(this.alarmPeriodInMinutes),
        treatMissingData: cloudwatch.TreatMissingData.MISSING,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD
      });

    this.cwAlarm.addAlarmAction(this.errorsAction);
    this.cwAlarm.addOkAction(this.errorsAction);

    // create a dashboard
    this.dashboard = new cloudwatch.Dashboard(this, 'JenkinsAgentFailures', {
      dashboardName: props.dashboardProps?.dashboardName ? props.dashboardProps.dashboardName : 'Jenkins_Agent_Failures', ...props.dashboardProps
    });

    this.dashboard.addWidgets(new cloudwatch.AlarmWidget({
      alarm: this.cwAlarm,
      title: this.cwAlarm.alarmName,
      height: 12,
      width: 12
    }));
  }
}
