#!/usr/bin/env groovy

package com.twitch.ipdev;

class AgentTests implements Serializable {
  def runAgentTests(script, Map testConfig, Map args) {
    // keys added to testConfig to support writing JUnit report
    // errorMessage, failureMessage, systemOut
    String[] failedTests = []
    Integer testCount = 0
    testConfig.each { thisTest ->
      testCount++
      if (args.get('debug', true).toBoolean()) {
        script.echo """Test Name: ${thisTest.key}
                      |  command: ${thisTest.value.command}""".stripMargin()
      }
      try {
        String[] result = runCmdOnNodeSavingExitCodeAndStdout(script, thisTest.value.command, 'Running Test: ' + thisTest.key)
        Integer rc = result[0].toInteger()
        String output = result[1]
        if (rc != 0) {
          throw new Exception(output)
        }
        if (thisTest.value.regex != null) {
          if (args.get('debug', true).toBoolean()) {
            script.echo "testing against regex: ${thisTest.value.regex}"
          }
          if (output =~ ~thisTest.value.regex){
            if (args.get('debug', true).toBoolean()) {
              script.echo "Success\n"
            }
          } else {
            thisTest.value.failureMessage = """Fail\n
                          |Received output: ${output}""".stripMargin()
            failedTests += thisTest.key
          }
        }
      } catch(Exception ex) {
        failedTests += thisTest.key
        thisTest.value.errorMessage = """Caught exception trying to execute test: ${thisTest.key}
                      |Received error: ${ex.toString()}\n""".stripMargin()
      }
    }

    script.echo "Executed ${testCount} tests"
    if (args.get('enableMetrics', true).toBoolean()) {
      script.echo 'Sending Cloudwatch Metrics'
      pushMetrics(script, failedTests.size(), args)
    }
    generateJUnitReport(script, testCount, failedTests.size(), testConfig, args.agentLabel)

    if (failedTests.size() > 0) {
      script.error "${failedTests.size()} tests failed:\n${failedTests.join(',')}\n"
    } else {
      script.echo "All tests passed successfully"
      return
    }
  }

  private void generateJUnitReport(script, Integer testCount, Integer failureCount, Map testConfig, String agentLabel) {
    script.echo "Generating JUnit report for '${agentLabel}'"
    String report = """<?xml version="1.0" ?>\n<testsuite name="${agentLabel}" tests="${testCount}" failures="${failureCount}" id="0" >\n"""
    report += "<properties>"
    // Loop through and write commands as properties
    testConfig.each { thisTest ->
      report += """<property name="${thisTest.key}" value="${thisTest.value.command}" />\n"""
    }
    report += "</properties>"

    // Now loop through and write the results
    testConfig.each { thisTest ->
      report += """<testcase name="${thisTest.key}">\n"""
      if (thisTest.value.containsKey('errorMessage')) {
        report += """<error message="${thisTest.value.errorMessage}"></error>\n"""
      }
      if (thisTest.value.containsKey('failureMessage')) {
        report += """<failure message="${thisTest.value.failureMessage}"></failure>\n"""
      }
      if (thisTest.value.containsKey('systemOut')) {
        report += """<system-out message="${thisTest.value.systemOut}"></system-out>\n"""
      }
      report += "</testcase>\n"
    }
    report += "</testsuite>\n"

    // Save the report
    script.writeFile(file: "tests/${agentLabel}.xml", text: report)

  }

  private void pushMetrics(script, Integer failureCount, Map args) {
    try {
      String cmd = "aws --region ${args.get('region', 'us-west-2')} cloudwatch put-metric-data" +
                    " --metric-name jenkins-agent-tests-${args.get('agentLabel')} --namespace JenkinsAgents --unit Count --value ${failureCount}" +
                    " --dimensions label=${args.get('agentLabel')}"
      if (args.get('debug', false).toBoolean()) {
        script.echo "Going to execute command:\n\n${cmd}\n\n"
      }

      String[] result = runCmdOnNodeSavingExitCodeAndStdout(script, cmd, 'Pushing Metrics')
      Integer rc = result[0].toInteger()
      String output = result[1]
      if (args.get('debug', false).toBoolean()) {
        script.echo "Metrics command output: ${output}"
      }

      if (rc != 0) {
        throw new Exception(output)
      }
    } catch(Exception ex) {
      // Leave this here just in case we really do throw an error
      script.echo """Caught exception. Failed to send metrics to Cloudwatch
                      |Received error: ${ex.toString()}\n""".stripMargin()
    }
  }

  /*  May not work if "cmd" already contains output redirection or more complex shell syntax. */
  private String[] runCmdOnNodeSavingExitCodeAndStdout(script, String cmd, String label) {
      Integer rc = 0
      String stdout = null
      String tempFilePath = "${script.env.WORKSPACE}/runCmdOnNodeSavingExitCodeAndStdout_" + UUID.randomUUID() + '.txt'

      if (script.isUnix()) {
          rc = script.sh(script: cmd + ' 2>&1 > ' + tempFilePath, returnStatus: true, encoding: 'UTF-8', label: label)
      } else {
          rc = script.bat(script: cmd + ' > ' + tempFilePath, returnStatus: true, encoding: 'UTF-8', label: label);
      }
      stdout = script.readFile(tempFilePath).trim()

      // Delete temporary file from the node
      if (script.isUnix()) {
          script.sh(script: 'rm -f ' + tempFilePath, returnStatus: true, encoding: 'UTF-8', label: 'Deleting tmp file')
      } else {
          script.bat(script: 'del /q ' + tempFilePath, returnStatus: true, encoding: 'UTF-8', label: 'Deleting tmp file');
      }

      return [ rc, stdout ]
  }
  private void writeJUnitResults() {

  }
}
