#!/usr/bin/env groovy
package com.twitch.ipdev

/**
 * Use github API to set status for any given commit
 */
class SetBuildStatus implements Serializable {
  // The state of the status. Can be one of error, failure, pending, or success.
  String state

  // GHE User to set status as
  String gheUser

  // GHE PAT of user
  String ghePAT

  // Required for this function to be wrapped by `withCredentials`
  String credentialsId

  // https://developer.github.com/v3/repos/statuses/
  // this URL should be in the format of - 
  // https://git-aws.internal.justin.tv/api/v3/repos/${GHE_ORG}/${GHE_REPO}/statuses/${GIT_COMMIT}
  String githubApiEndpoint

  // A short description of the status.
  String description = "Flexo Builds"

  // The target URL to associate with this status. This URL will be linked from the GitHub UI to allow users to easily see the source of the status.
  // For example, if your continuous integration system is posting build status, you would want to provide the deep link for the build output for this specific SHA:
  String targetUrl = ""

  // A string label to differentiate this status from the status of other systems
  String context = "default"

  // Override to true for more debug output
  boolean debug = false

  def setStatus() {
    if(this.debug){
      println("SetBuildStatus.groovy - State: ${this.state}")
      println("SetBuildStatus.groovy - Description: ${this.description}")
      println("SetBuildStatus.groovy - TargetUrl: ${this.targetUrl}")
      println("SetBuildStatus.groovy - Context: ${this.context}")
      println("SetBuildStatus.groovy - GitApiUrl: ${this.githubApiEndpoint}")
      println("SetBuildStatus.groovy - Debug: ${this.debug}")
      println("SetBuildStatus.groovy - gheUser: ${this.gheUser}")
      println("SetBuildStatus.groovy - credentialsId: ${credentialsId}")
    }

    // Groovy-fu to create an HTTP request
    def post = new URL(this.githubApiEndpoint).openConnection();
    
    // Create payload
    def data = [
      state: this.state,
      description: this.description,
      target_url: this.targetUrl,
      context: this.context
    ]
    
    post.setRequestMethod("POST")
    post.setDoOutput(true)
    post.setRequestProperty("Content-Type", "application/json")

    // Set up Authorization with GHE user/PAT
    post.setRequestProperty("Authorization",
      "Basic " + new String(Base64.getEncoder().encode("${this.gheUser}:${this.ghePAT}".getBytes())))
    
    // Execute
    post.getOutputStream().write(groovy.json.JsonOutput.toJson(data).getBytes("UTF-8"))

    // If we don't get the response code, this seems to exit too quickly and the 
    // status might not get updated on GHE, so don't touch this line - thar be dragons
    def postRC = post.getResponseCode();

    // If debug is turned on, pop out a little extra info
    if(this.debug){
      println("SetBuildStatus.groovy ${postRC}- ${post.getInputStream().getText()}")
    }
  }
}
