package groovy

import com.twitch.ipdev.flexo.FlexoPipelineStage
import groovy.libs.FlexoPipelineTest
import org.junit.*

import static groovy.test.GroovyAssert.assertNotNull
import static org.junit.Assert.assertEquals

class FlexoPipelineStageTest extends FlexoPipelineTest {
    Script flexo

    FlexoPipelineStage stageDefault
    FlexoPipelineStage stageCustom
    FlexoPipelineStage stageFailure

    @Before
    void setUp() {
        super.setUp()
        flexo = loadScript("vars/flexo.groovy")

        def env = binding.getVariable("env")

        // Mock the setBuildStatus*() functions used for GitHub status reporting
        helper.registerAllowedMethod("setBuildStatusPending", [[:].getClass()], {})
        helper.registerAllowedMethod("setBuildStatusSuccess", [[:].getClass()], {})
        helper.registerAllowedMethod("setBuildStatusFailure", [[:].getClass()], {})

        helper.registerAllowedMethod("sh", [String.class], {cmd->
            if (cmd.contains('breakme')) {
                binding.getVariable('currentBuild').result = 'FAILURE'
                return false
            }
        })

        stageDefault = new FlexoPipelineStage(flexo, com.twitch.ipdev.flexo.FlexoStage.BUILD, true, null, env)
        stageCustom = new FlexoPipelineStage(flexo, com.twitch.ipdev.flexo.FlexoStage.LINT, false, { echo 'test stage' }, env)
        stageFailure = new FlexoPipelineStage(flexo, com.twitch.ipdev.flexo.FlexoStage.UNIT_TEST, true, { sh 'breakme' }, env)
    }

    @Test
    void testPipelineStageReportStatusOverride() {
        // should be true for stageDefault
        assert stageDefault.reportStatus
        // should be false for stageCustom1
        assert !stageCustom.reportStatus
    }

    @Test
    void testPipelineStageClosureDelegation() {
        // when passed a null commands Closure, we set a default
        assertNotNull stageDefault.commands
        // the constructor should have set the delegate and called setResolveStrategy
        assert stageDefault.commands.getResolveStrategy() == Closure.DELEGATE_FIRST
        assert stageDefault.commands.getDelegate() == flexo
    }

    @Test
    void testPipelineStageRun() {
        stageDefault.run()
        assertJobStatusSuccess()

        def buildStatusArgs = "{debug=true, description=Flexo: ${stageDefault.type.label}, credentialsId=flexo-org-service_pat}"
        // should have called setBuildStatusPending exactly once
        assertEquals methodCallCount("setBuildStatusPending"), 1
        assert methodCalledWithArgs("setBuildStatusPending", buildStatusArgs)

        // should have called setBuildStatusSuccess exactly once
        assertEquals methodCallCount("setBuildStatusSuccess"), 1
        assert methodCalledWithArgs("setBuildStatusSuccess", buildStatusArgs)

        // should not have called setBuildStatusFailure
        assertEquals methodCallCount("setBuildStatusFailure"), 0
    }

    @Test
    void testPipelineStageFailureRun() {
        stageFailure.run()
        assertJobStatusFailure()

        def buildStatusArgs = "{debug=true, description=Flexo: ${stageFailure.type.label}, credentialsId=flexo-org-service_pat}"
        // should have called setBuildStatusPending exactly once
        assertEquals methodCallCount("setBuildStatusPending"), 1
        assert methodCalledWithArgs("setBuildStatusPending", buildStatusArgs)

        // TODO: how can we test this fail case?
        // mock execution doesn't properly handle Exceptions and failures

        // should have called setBuildStatusFailure exactly once
        // assertEquals methodCallCount("setBuildStatusFailure"), 1
        // assert methodCalledWithArgs("setBuildStatusFailure", buildStatusArgs)

        // should not have called setBuildStatusSuccess
        // assertEquals methodCallCount("setBuildStatusSuccess"), 0
    }

    @Test
    void testPipelineStageRunNoStatusReport() {
        stageCustom.run()
        assertJobStatusSuccess()

        // should not have called setBuildStatusPending
        assertEquals methodCallCount("setBuildStatusPending"), 0
        // should not have called setBuildStatusSuccess
        assertEquals methodCallCount("setBuildStatusSuccess"), 0
        // should not have called setBuildStatusFailure
        assertEquals methodCallCount("setBuildStatusFailure"), 0
    }
}
