package groovy

import com.twitch.ipdev.SetBuildStatus
import org.junit.*
import com.lesfurets.jenkins.unit.*
import com.sun.net.httpserver.*
import static com.lesfurets.jenkins.unit.MethodCall.callArgsToString
import static org.assertj.core.api.Assertions.assertThat

/**
 * Test the vars/setBuildStatus*.groovy and src/com.twitch.ipdev/SetBuildStatus functions
 *
 * Note from Richard:
 *   - Directly loading the setBuildStatusSuccess/setBuildStatusFailure/setBuildStatusPending
 *       functions proved to be difficult since they call setBuildStatusCommon() and I cannot
 *       figure out for the life of me the arcane incantation to get setBuildStatusCommon()
 *       loaded into helper.registerAllowedMethod(), thus - the tests imitate the functionality
 *       of those functions as a "best effort" test implementation
 */
class setBuildStatusTest extends BasePipelineTest {
    Script setBuildStatusCommon
    int port
    InetSocketAddress socket
    HttpServer httpServer

    // these git vars can be anything, just using a real commit because it makes me feel good
    String gitCommit = '9d51aebb8f0d85d616236fdb1e708a67a50c9500'
    String gitOrg = 'flexo'
    String gitRepo = 'flexo-libs'

    @Before
    void setUp() {
        super.setUp()

        // Find an unused port, take it and then immediately release it for use with the HTTP server
        // The internet is failing me for a better way to do this without writing a really janky for loop
        def s = new ServerSocket(0)
        port = s.getLocalPort()
        s.close()

        // Create a local HTTP server to deal with our library's requests
        // Not my first choice to do this, but BasePipelineTest and using Mockito/PowerMockito
        //    to mock Url().openConnection() was throwing errors that aren't worth our time to figure out
        socket = new InetSocketAddress(port)
        httpServer = HttpServer.create(socket, 0)
        httpServer.createContext("/", new HttpHandler() {
            void handle(HttpExchange exchange) throws IOException {
                byte[] response = "{\"success\": true}".getBytes()
                exchange.sendResponseHeaders(HttpURLConnection.HTTP_OK, response.length)
                exchange.getResponseBody().write(response)
                exchange.close()
            }
        })
        httpServer.createContext("/${gitOrg}/${gitRepo}", new HttpHandler() {
            void handle(HttpExchange exchange) throws IOException {
                byte[] response = "{\"success\": true}".getBytes()
                exchange.sendResponseHeaders(HttpURLConnection.HTTP_OK, response.length)
                exchange.getResponseBody().write(response)
                exchange.close()
            }
        })
        httpServer.start()

        // Load the script we're testing
        setBuildStatusCommon = loadScript("vars/setBuildStatusCommon.groovy")

        // Mock all the environment and credential variables that Jenkins sets up for us normally
        def tempEnv = binding.getVariable("env")
        tempEnv.GIT_COMMIT  = gitCommit
        binding.setVariable("env", tempEnv)
        binding.setVariable('GHE_USER', 'user')
        binding.setVariable('GHE_PAT', 'patpatpat')
        binding.setVariable('credentialsId', 'credId')

        setBuildStatusCommon.setBinding(binding)

        // Mock the scm step
        helper.registerAllowedMethod("scm",  [Map.class], { ->
            String url = "http://127.0.0.1/${gitOrg}/${gitRepo}.git"
        })

        // Mock the withCredentials.usernamePassword() step
        helper.registerAllowedMethod("usernamePassword", [[:].getClass()], {})
    }

    def testSetBuildStatusCommonForTest(Map args=[:]) {
        setBuildStatusCommon(
                context: "test",
                state: args.state,
                githubApiEndpoint: "http://127.0.0.1:${port}/api/v3/repos/${gitOrg}/${gitRepo}/statuses/${gitCommit}",
                debug: true
        )

        assertThat(helper.callStack.findAll { call ->
            call.methodName == "usernamePassword"
        }.any { call ->
            callArgsToString(call).contains("credentialsId=null, usernameVariable=GHE_USER, passwordVariable=GHE_PAT")
        }).isTrue()

        assertThat(helper.callStack.findAll { call ->
            call.methodName == "withCredentials"
        }.any { call ->
            callArgsToString(call).contains("null")
        }).isTrue()
    }

    @Test
    void testSetBuildStatusSuccess() {
        testSetBuildStatusCommonForTest([state: 'success'])
        assertJobStatusSuccess()
    }

    @Test
    void testSetBuildStatusFailure() {
        testSetBuildStatusCommonForTest([state: 'failure'])
        assertJobStatusSuccess()
    }

    @Test
    void testSetBuildStatusPending() {
        testSetBuildStatusCommonForTest([state: 'pending'])
        assertJobStatusSuccess()
    }

    @Test
    void testSetBuildStatus() {
        new SetBuildStatus([
            gheUser: 'user',
            ghePAT: 'patpatpat',
            credentialsId: 'credId',
            context: "test",
            state: 'success',
            githubApiEndpoint: "http://127.0.0.1:${port}/api/v3/repos/${gitOrg}/${gitRepo}/statuses/${gitCommit}",
            debug: true
        ]).setStatus()
        assertJobStatusSuccess()
    }

    @After
    void tearDown(){
        // Kill the HTTP server
        httpServer.stop(0)
    }
}

