#!/usr/bin/env ruby
require 'aws-sdk'
require 'json'

require_relative './patches'
require_relative './constants'
require_relative './helpers'

task_family = TASK_FAMILY || TEAMCITY_PROJECT_NAME

log "Updating the task definition `#{task_family}` to use #{IMAGE_VERSION}"

DEPLOY_ENV.each do |nickname|
  aws_region = AWS_REGIONS[nickname]

  ecs = authenticate account: DEPLOY_AWS_ACCOUNT, region: aws_region, assume_role: DEPLOY_ASSUME_ROLE,
                     session_name: "updating-#{task_family}-to-#{IMAGE_VERSION}"

  results = ecs.register_task_definition \
    family: task_family,
    task_role_arn: "arn:aws:iam::#{DEPLOY_AWS_ACCOUNT}:role/#{regionalized_role(TASK_ROLE, aws_region)}",
    container_definitions: [
      {
        name: task_family,
        essential: true,
        image: "#{AWS_ECR_URL}/#{TEAMCITY_PROJECT_NAME}:#{IMAGE_VERSION}",
        memory: 1024,
        environment: [
          { name: 'ASPNETCORE_ENVIRONMENT', value: ASPNETCORE_ENVIRONMENT },
          # Service Info Start
          { name: 'ServiceInfo__AWS_ECR_URL', value: AWS_ECR_URL },
          { name: 'ServiceInfo__CLUSTER_NAME', value: CLUSTER_NAME },
          { name: 'ServiceInfo__DEPLOY_AWS_ACCOUNT', value: DEPLOY_AWS_ACCOUNT },
          { name: 'ServiceInfo__DEPLOY_CLOUDWATCH_REGION', value: DEPLOY_CLOUDWATCH_REGION },
          { name: 'ServiceInfo__DESIRED_COUNT', value: DESIRED_COUNT },
          { name: 'ServiceInfo__IMAGE_VERSION', value: IMAGE_VERSION },
          { name: 'ServiceInfo__SERVICE_NAME', value: SERVICE_NAME },
          { name: 'ServiceInfo__TARGET_ENV', value: TARGET_ENV },
          { name: 'ServiceInfo__TASK_FAMILY', value: task_family },
          { name: 'ServiceInfo__TEAMCITY_PROJECT_NAME', value: TEAMCITY_PROJECT_NAME },
          { name: 'ServiceInfo__REGION_AIRPORT_CODE', value: nickname },
          { name: 'ServiceInfo__REGION_AWS', value: aws_region },
          # Service Info End
          # Monitoring Start
          { name: 'Monitoring__AlertHandler', value: Monitoring__AlertHandler },
          { name: 'Monitoring__HeartbeatHandler', value: Monitoring__HeartbeatHandler },
          { name: 'Monitoring__KeepalivePagerTeam', value: Monitoring__KeepalivePagerTeam },
          { name: 'Monitoring__MetricHandlers', value: Monitoring__MetricHandlers },
          { name: 'Monitoring__SensuSnsTopic', value: Monitoring__SensuSnsTopic.gsub('us-west-2', aws_region) },
          # Monitoring End
          { name: 'Arcana__DefaultBucket', value: Arcana__DefaultBucket },
          { name: 'Arcana__DefaultService', value: Arcana__DefaultService },
          { name: 'AuditLog__S3__BucketName', value: AuditLog__S3__BucketName },
          { name: 'AuditLog__SNS__TopicArn', value: AuditLog__SNS__TopicArn },
          { name: 'Cors__Origins', value: Cors__Origins },
          { name: 'Elastic__Host', value: Elastic__Host },
          { name: 'Elastic__IndexNameOverride', value: Elastic__IndexNameOverride },
          { name: 'Elastic__TimeSeriesType', value: Elastic__TimeSeriesType },
          { name: 'Elastic__ShardCountOverride', value: Elastic__ShardCountOverride },
          { name: 'Elastic__ReplicaCountOverride', value: Elastic__ReplicaCountOverride },
          { name: 'Logging__Service__Url', value: regionalized_service(Logging__Service__Url, nickname) },
          { name: 'Sqs__WorkerConfiguration__AuditMessages__QueueUrl', value: Sqs__WorkerConfiguration__AuditMessages__QueueUrl },
          { name: 'Sqs__WorkerConfiguration__AuditMessages__SubscriptionConfirmationTopicArnsWhitelist', value: Sqs__WorkerConfiguration__AuditMessages__SubscriptionConfirmationTopicArnsWhitelist },
          { name: 'Sqs__WorkerConfiguration__AuditMessages__NumberOfWorkers', value: Sqs__WorkerConfiguration__AuditMessages__NumberOfWorkers }
        ],
        port_mappings: [
          { host_port: 0, container_port: 8080, protocol: 'tcp' },
          { host_port: 0, container_port: 5000, protocol: 'tcp' }
        ],
        log_configuration: {
          log_driver: 'awslogs',
          options: {
            'awslogs-group' => task_family,
            'awslogs-region' => DEPLOY_CLOUDWATCH_REGION,
            'awslogs-stream-prefix' => "#{task_family}-#{nickname}"
          }
        }
      }
    ]

  log "Update complete in #{nickname}. API Output:"
  log JSON.pretty_unparse(results.to_hash)

  task_revision = results.task_definition.revision

  log "Updating env.TASK_REVISION_#{nickname.upcase} to `#{task_revision}`"
  set_param "env.TASK_REVISION_#{nickname.upcase}", task_revision
end
