﻿using Moq;
using System;
using System.Collections.Generic;
using System.Threading;
using System.Threading.Tasks;
using Amazon.SQS.Model;
using FluentAssertions;
using Microsoft.Extensions.Logging.Abstractions;
using Newtonsoft.Json;
using Twitch.AuditLogService.ChangeLog.Models;
using Twitch.AuditLogService.Models;
using Twitch.Shared.Sqs.Models;
using Xunit;

namespace Twitch.AuditLogService.Tests
{
    [Trait("Category", "Unit")]
    public class AuditLogMessageProcessorTests : IDisposable
    {
        private const string AuditLogEntryGuid = "unique_auditlog_entry_id";

        private readonly Message _sqsMessage = new Message { Body = "random message " };
        private readonly Message _snsSubscriptionConfirmationMessage = new Message { Body = @"{ ""Type"":""SubscriptionConfirmation"", ""SigningCertURL"":""https://sns"" } " };

        private readonly Message _snsAuditLogNotificationMessage = new Message
        {
            Body = JsonConvert.SerializeObject(
                new
                {
                    Type = "Notification",
                    SigningCertURL = "https://sns",
                    Message = JsonConvert.SerializeObject(new AuditLogElasticEntry { 
                        Created = 1562632940414, 
                        Id = AuditLogEntryGuid, 
                        DataChanges = new Dictionary<string, AuditLogElasticDataChange>{ {"test", new AuditLogElasticDataChange
                        {
                            Before = "a",
                            After = "b",
                            Key = "test"
                        }}}
                    })
                }
            )
        };

        private readonly Message _snsChangeLogNotificationMessage = new Message
        {
            Body = JsonConvert.SerializeObject(
                new
                {
                    Type = "Notification",
                    SigningCertURL = "https://sns",
                    Message = JsonConvert.SerializeObject(new ChangeLogElasticEntry { Timestamp = 1562632940 })
                }
            )
        };

        private readonly MockRepository _mockRepository;
        private readonly Mock<IAuditLogSearchManager> _mockAuditLogSearchManager;

        public AuditLogMessageProcessorTests()
        {
            _mockRepository = new MockRepository(MockBehavior.Strict);
            _mockAuditLogSearchManager = _mockRepository.Create<IAuditLogSearchManager>();
        }

        public void Dispose()
        {
            _mockRepository.VerifyAll();
        }

        private AuditLogMessageProcessor CreateAuditLogMessageProcessor()
        {
            return new AuditLogMessageProcessor(
                new NullLogger<AuditLogMessageProcessor>(), 
                _mockAuditLogSearchManager.Object);
        }

        [Fact]
        public async Task TryProcessMessagesAsync_NonSnsMessageGiven_ShouldNotProcessMessage()
        {
            // Arrange
            var auditLogMessageProcessor = CreateAuditLogMessageProcessor();
            var messages = new []{ WrappedSqsMessage.CreateFromMessage(_sqsMessage) };

            // Act
            var result = await auditLogMessageProcessor.TryProcessMessagesAsync(messages, CancellationToken.None);

            // Assert
            result.Should().BeEmpty();
        }

        [Fact]
        public async Task TryProcessMessagesAsync_SnsSubscriptionConfirmationMessageGiven_ShouldNotProcessMessage()
        {
            // Arrange
            var auditLogMessageProcessor = CreateAuditLogMessageProcessor();
            var messages = new []{ WrappedSqsMessage.CreateFromMessage(_snsSubscriptionConfirmationMessage) };

            // Act
            var result = await auditLogMessageProcessor.TryProcessMessagesAsync(messages, CancellationToken.None);

            // Assert
            result.Should().BeEmpty();
        }

        [Fact]
        public async Task TryProcessMessagesAsync_SnsChangeLogNotificationMessageGiven_ShouldNotProcessMessage()
        {
            // Arrange
            var auditLogMessageProcessor = CreateAuditLogMessageProcessor();
            var messages = new []{ WrappedSqsMessage.CreateFromMessage(_snsChangeLogNotificationMessage) };

            // Act
            var result = await auditLogMessageProcessor.TryProcessMessagesAsync(messages, CancellationToken.None);

            // Assert
            result.Should().BeEmpty();
        }

        [Fact]
        public async Task TryProcessMessagesAsync_SnsAuditLogNotificationMessageGiven_ShouldProcessMessage()
        {
            // Arrange
            var auditLogMessageProcessor = CreateAuditLogMessageProcessor();
            var messages = new []{ WrappedSqsMessage.CreateFromMessage(_snsAuditLogNotificationMessage) };
            _mockAuditLogSearchManager.Setup(sm => sm.IndexAsync(It.Is<AuditLogElasticEntry>(entry => entry.Id.Equals(AuditLogEntryGuid, StringComparison.Ordinal)))).ReturnsAsync(true);

            // Act
            var result = await auditLogMessageProcessor.TryProcessMessagesAsync(messages, CancellationToken.None);

            // Assert
            // ReSharper disable once CoVariantArrayConversion
            result.Should().BeEquivalentTo(messages);
        }

        [Fact]
        public async Task TryProcessMessagesAsync_IndexAsyncThrowsExceptionGiven_ShouldNotThrow()
        {
            // Arrange
            var auditLogMessageProcessor = CreateAuditLogMessageProcessor();
            var messages = new []{ WrappedSqsMessage.CreateFromMessage(_snsAuditLogNotificationMessage) };
            _mockAuditLogSearchManager.Setup(sm => sm.IndexAsync(It.IsAny<AuditLogElasticEntry>())).Throws<InvalidOperationException>();

            // Act
            var result = await auditLogMessageProcessor.TryProcessMessagesAsync(messages, CancellationToken.None);

            // Assert
            result.Should().BeEmpty();
        }
    }
}
