﻿using Microsoft.Extensions.Caching.Memory;
using Microsoft.Extensions.Options;
using Moq;
using Nest;
using System;
using System.Collections.Generic;
using System.Linq.Expressions;
using System.Threading;
using System.Threading.Tasks;
using Elasticsearch.Net;
using FluentAssertions;
using Microsoft.Extensions.Logging.Abstractions;
using Moq.Language.Flow;
using Newtonsoft.Json.Linq;
using Twitch.AuditLogService.Models;
using Twitch.ElasticSearch;
using Twitch.ElasticSearch.Search;
using Xunit;

namespace Twitch.AuditLogService.Tests
{
    [Trait("Category", "Unit")]
    public class AuditLogSearchManagerTests : IDisposable
    {
        private readonly MockRepository _mockRepository;

        private readonly Mock<IElasticClient> _mockElasticClient;
        private readonly Mock<IOptions<SearchConfiguration>> _mockOptions;
        private readonly Mock<IIndexResponse> _indexResponseMock;

        public AuditLogSearchManagerTests()
        {
            _mockRepository = new MockRepository(MockBehavior.Strict);

            _mockElasticClient = _mockRepository.Create<IElasticClient>();
            _mockOptions = _mockRepository.Create<IOptions<SearchConfiguration>>();
            _mockOptions.SetupGet(o => o.Value).Returns(new SearchConfiguration { TimeSeriesType = TimeSeriesIndexType.Monthly });

            var apiCallDetailsMock = _mockRepository.Create<IApiCallDetails>();
            apiCallDetailsMock.SetupGet(e => e.Success).Returns(true);

            _indexResponseMock = _mockRepository.Create<IIndexResponse>();

            var existsResponseMock = _mockRepository.Create<IExistsResponse>();
            existsResponseMock.SetupGet(e => e.Exists).Returns(true);
            existsResponseMock.SetupGet(e => e.ApiCall).Returns(apiCallDetailsMock.Object);

            _mockElasticClient.Setup(m => m.IndexTemplateExists(It.IsAny<Name>(), null)).Returns(existsResponseMock.Object);
        }

        public void Dispose()
        {
            _mockRepository.VerifyAll();
        }

        private AuditLogSearchManager CreateAuditLogSearchManager()
        {
            return new AuditLogSearchManager(
                _mockElasticClient.Object,
                _mockOptions.Object,
                new NullLogger<AuditLogSearchManager>(),
                new MemoryCache(new MemoryCacheOptions()));
        }

        private readonly Expression<Func<IElasticClient, Task<ISearchResponse<AuditLogElasticEntry>>>> _searchExpression = ec =>
            ec.SearchAsync<AuditLogElasticEntry>(It.IsAny<ISearchRequest>(), It.IsAny<CancellationToken>());
        private IReturnsResult<IElasticClient> ConfigureMockSearchResponse(AuditLogElasticEntry[] mockEntries, bool validResponse = true)
        {
            var searchResponseMock = new Mock<ISearchResponse<AuditLogElasticEntry>>();
            searchResponseMock.SetupGet(x => x.IsValid).Returns(validResponse);
            searchResponseMock.Setup(x => x.Documents).Returns(mockEntries);
            return _mockElasticClient.Setup(_searchExpression).ReturnsAsync(searchResponseMock.Object);
        }

        private IReturnsResult<IElasticClient> ConfigureMockTermFilterResponse(Dictionary<string, Dictionary<string, long>> mockTermFilters, bool validResponse = true)
        {
            var results = new Dictionary<string, IAggregate>();
            foreach (var mockTermFilter in mockTermFilters)
            {
                var innerAggs = new List<IBucket>();
                foreach (var term in mockTermFilter.Value)
                {
                    innerAggs.Add(new KeyedBucket<object> { Key = term.Key, DocCount = term.Value });
                }

                var agg = new BucketAggregate
                {
                    DocCount = mockTermFilter.Value.Count,
                    Items = innerAggs
                };
                results[mockTermFilter.Key] = agg;
            }
            var searchResponseMock = new Mock<ISearchResponse<AuditLogElasticEntry>>();
            searchResponseMock.SetupGet(x => x.IsValid).Returns(validResponse);
            searchResponseMock.SetupGet(x => x.Aggregations).Returns(results);
            searchResponseMock.SetupGet(x => x.Aggs).Returns(new AggregationsHelper(results));
            return _mockElasticClient.Setup(_searchExpression).ReturnsAsync(searchResponseMock.Object);
        }

        [Fact]
        public async Task IndexAsync_ValidItemGiven_ShouldIndexItem()
        {
            // Arrange
            var item = new AuditLogElasticEntry { Created = DateTimeOffset.UtcNow.ToUnixTimeMilliseconds(), Id = Guid.NewGuid().ToString() };
            Expression<Func<IElasticClient, Task<IIndexResponse>>> indexExpression = ec =>
                ec.IndexAsync(It.Is<IndexRequest<AuditLogElasticEntry>>(ir => ir.Document == item), null, It.IsAny<CancellationToken>());
            _indexResponseMock.SetupGet(ir => ir.IsValid).Returns(true);
            _mockElasticClient.Setup(indexExpression).ReturnsAsync(_indexResponseMock.Object);
            var auditLogSearchManager = CreateAuditLogSearchManager();

            // Act
            var result = await auditLogSearchManager.IndexAsync(item);

            // Assert
            result.Should().BeTrue();
            _mockElasticClient.Verify(indexExpression, Times.Once);
        }

        [Fact]
        public async Task IndexAsync_NullItemGiven_ShouldThrowArgumentNullException()
        {
            // Arrange
            var auditLogSearchManager = CreateAuditLogSearchManager();

            // Act
            Func<Task<bool>> action = async () => await auditLogSearchManager.IndexAsync(null);

            // Assert
            await action.Should().ThrowAsync<ArgumentNullException>();
        }

        [Fact]
        public async Task IndexAsync_InvalidResponseGiven_ShouldReturnFalse()
        {
            // Arrange
            var item = new AuditLogElasticEntry { Created = DateTimeOffset.UtcNow.ToUnixTimeMilliseconds(), Id = Guid.NewGuid().ToString() };
            _indexResponseMock.SetupGet(ir => ir.IsValid).Returns(false);
            _indexResponseMock.SetupGet(ir => ir.OriginalException).Returns(new Exception());
            _indexResponseMock.SetupGet(ir => ir.DebugInformation).Returns("test failure");
            _mockElasticClient.Setup(m => m.IndexAsync(It.Is<IndexRequest<AuditLogElasticEntry>>(ir => ir.Document == item), null, It.IsAny<CancellationToken>()))
                .ReturnsAsync(_indexResponseMock.Object);
            var auditLogSearchManager = CreateAuditLogSearchManager();

            // Act
            var result = await auditLogSearchManager.IndexAsync(item);

            // Assert
            result.Should().BeFalse();
        }

        [Fact]
        public async Task SearchAsync_ScrollingSearchFilterGiven_ShouldConstructCorrectQuery()
        {
            // Arrange
            var unixTimeMilliseconds = DateTimeOffset.UtcNow.ToUnixTimeMilliseconds();
            var guid = Guid.NewGuid().ToString();
            var mockEntries = new[] { new AuditLogElasticEntry { Created = unixTimeMilliseconds, Id = guid } };
            SearchDescriptor<AuditLogElasticEntry> searchDescriptor = null;
            ConfigureMockSearchResponse(mockEntries).Callback((ISearchRequest request, CancellationToken token) => { searchDescriptor = request as SearchDescriptor<AuditLogElasticEntry>; });

            var auditLogSearchManager = CreateAuditLogSearchManager();
            AuditLogSearch searchCriteria = new AuditLogSearch
            {
                SearchFilter = new ScrollingSearchFilter
                {
                    Direction = SearchDirection.Newer,
                    Timestamp = unixTimeMilliseconds,
                    Id = guid
                }
            };
            int pageSize = 500;

            // Act
            var result = await auditLogSearchManager.SearchAsync(searchCriteria, pageSize);

            // Assert
            // ReSharper disable once CoVariantArrayConversion
            result.Should().BeEquivalentTo(mockEntries);
            _mockElasticClient.Verify(_searchExpression, Times.Once);

            // Assert Correct Search Descriptor Behavior
            var searchToken = JToken.Parse(new ElasticClient().Serializer.SerializeToString(searchDescriptor));
            searchToken.Value<int>("size").Should().Be(pageSize);
            searchToken.SelectToken("$.sort..created").Value<string>("order").Should().Be("asc");
            searchToken.SelectToken("$.sort..id").Value<string>("order").Should().Be("desc");
            searchToken.SelectToken("search_after").Values<string>().Should().ContainInOrder(unixTimeMilliseconds.ToString(), guid);
        }

        [Fact]
        public async Task SearchAsync_SearchQueryGiven_ShouldConstructCorrectQuery()
        {
            // Arrange
            var handler = "TestHandler";
            var mockEntries = new[] { new AuditLogElasticEntry { Created = DateTimeOffset.UtcNow.ToUnixTimeMilliseconds(), Id = Guid.NewGuid().ToString() } };
            SearchDescriptor<AuditLogElasticEntry> searchDescriptor = null;
            ConfigureMockSearchResponse(mockEntries).Callback((ISearchRequest request, CancellationToken token) => { searchDescriptor = request as SearchDescriptor<AuditLogElasticEntry>; });

            var auditLogSearchManager = CreateAuditLogSearchManager();
            AuditLogSearch searchCriteria = new AuditLogSearch
            {
                Query = new AuditLogSearchQuery
                {
                    Handler = handler
                }
            };
            int pageSize = 500;

            // Act
            var result = await auditLogSearchManager.SearchAsync(searchCriteria, pageSize);

            // Assert
            // ReSharper disable once CoVariantArrayConversion
            result.Should().BeEquivalentTo(mockEntries);
            _mockElasticClient.Verify(_searchExpression, Times.Once);

            // Assert Correct Search Descriptor Behavior
            var searchToken = JToken.Parse(new ElasticClient().Serializer.SerializeToString(searchDescriptor));
            var handlerToken = searchToken.SelectToken("query.match.handler", true);
            handlerToken.Value<string>("query").Should().Be(handler);
        }

        [Fact]
        public async Task SearchAsync_SearchFilterGiven_ShouldConstructCorrectQuery()
        {
            // Arrange
            var mockEntries = new[] { new AuditLogElasticEntry { Created = DateTimeOffset.UtcNow.ToUnixTimeMilliseconds(), Id = Guid.NewGuid().ToString() } };
            SearchDescriptor<AuditLogElasticEntry> searchDescriptor = null;
            ConfigureMockSearchResponse(mockEntries).Callback((ISearchRequest request, CancellationToken token) => { searchDescriptor = request as SearchDescriptor<AuditLogElasticEntry>; });

            var auditLogSearchManager = CreateAuditLogSearchManager();
            AuditLogSearch searchCriteria = new AuditLogSearch
            {
                Filter = new AuditLogSearchFilter
                {
                    ActionType = new [] { "read", "write" },
                    Name = new [] { "name1", "name2", "name3" }
                }
            };
            int pageSize = 500;

            // Act
            var result = await auditLogSearchManager.SearchAsync(searchCriteria, pageSize);

            // Assert
            // ReSharper disable once CoVariantArrayConversion
            result.Should().BeEquivalentTo(mockEntries);
            _mockElasticClient.Verify(_searchExpression, Times.Once);

            // Assert Correct Search Descriptor Behavior
            var searchToken = JToken.Parse(new ElasticClient().Serializer.SerializeToString(searchDescriptor));
            var filterToken = searchToken.SelectToken("$.query.bool.filter", true);
            filterToken.SelectToken("..terms.actionType").Values<string>().Should().BeEquivalentTo(searchCriteria.Filter.ActionType);
            filterToken.SelectToken("..terms.name").Values<string>().Should().BeEquivalentTo(searchCriteria.Filter.Name);
        }

        // Note by Felix Kastner (fkastne):
        // This test is annoyingly large, but it is important to test that multiple filter conditions can be combined.
        [Fact]
        public async Task SearchAsync_AllSearchOptionsGiven_ShouldConstructCorrectQuery()
        {
            // Arrange
            var unixTimeMilliseconds = DateTimeOffset.UtcNow.ToUnixTimeMilliseconds();
            var guid = Guid.NewGuid().ToString();
            var handler = "TestHandler";
            var mockEntries = new[] { new AuditLogElasticEntry { Created = unixTimeMilliseconds, Id = guid } };
            SearchDescriptor<AuditLogElasticEntry> searchDescriptor = null;
            ConfigureMockSearchResponse(mockEntries).Callback((ISearchRequest request, CancellationToken token) => { searchDescriptor = request as SearchDescriptor<AuditLogElasticEntry>; });

            var auditLogSearchManager = CreateAuditLogSearchManager();
            AuditLogSearch searchCriteria = new AuditLogSearch
            {
                SearchFilter = new ScrollingSearchFilter
                {
                    Direction = SearchDirection.Newer,
                    Timestamp = unixTimeMilliseconds,
                    Id = guid
                },
                Query = new AuditLogSearchQuery
                {
                    Handler = handler
                },
                Filter = new AuditLogSearchFilter
                {
                    ActionType = new[] { "read", "write" },
                    Name = new[] { "name1", "name2", "name3" }
                }
            };
            int pageSize = 500;

            // Act
            var result = await auditLogSearchManager.SearchAsync(searchCriteria, pageSize);

            // Assert
            // ReSharper disable once CoVariantArrayConversion
            result.Should().BeEquivalentTo(mockEntries);
            _mockElasticClient.Verify(_searchExpression, Times.Once);

            // Assert Correct Search Descriptor Behavior
            var searchToken = JToken.Parse(new ElasticClient().Serializer.SerializeToString(searchDescriptor));
            searchToken.Value<int>("size").Should().Be(pageSize);

            // Assert that Search After was correctly applied:
            searchToken.SelectToken("$.sort..created").Value<string>("order").Should().Be("asc");
            searchToken.SelectToken("$.sort..id").Value<string>("order").Should().Be("desc");
            searchToken.SelectToken("search_after").Values<string>().Should().ContainInOrder(unixTimeMilliseconds.ToString(), guid);

            // Assert that Text Search was correctly applied:
            var handlerToken = searchToken.SelectToken("query.bool.must[0].match.handler", true);
            handlerToken.Value<string>("query").Should().Be(handler);

            // Assert that Term Filters were correctly applied:
            var filterToken = searchToken.SelectToken("$.query.bool.filter", true);
            filterToken.SelectToken("..terms.actionType").Values<string>().Should().BeEquivalentTo(searchCriteria.Filter.ActionType);
            filterToken.SelectToken("..terms.name").Values<string>().Should().BeEquivalentTo(searchCriteria.Filter.Name);
        }

        [Fact]
        public async Task SearchAsync_InvalidResponseGiven_ShouldReturnEmptyCollection()
        {
            // Arrange
            var mockEntries = new[] { new AuditLogElasticEntry { Created = DateTimeOffset.UtcNow.ToUnixTimeMilliseconds(), Id = Guid.NewGuid().ToString() } };
            ConfigureMockSearchResponse(mockEntries, false);
            var auditLogSearchManager = CreateAuditLogSearchManager();

            // Act
            var result = await auditLogSearchManager.SearchAsync(new AuditLogSearch(), 500);

            // Assert
            result.Should().BeEmpty();
        }


        [Fact]
        public async Task GetTermFiltersAsync_ShouldConstructCorrectQuery()
        {
            // Arrange
            var auditLogSearchManager = CreateAuditLogSearchManager();
            var mockTermFilters = new Dictionary<string, Dictionary<string, long>>
            {
                { "Source", new Dictionary<string, long> { { "TestName1", 5 } } },
                { "Username", new Dictionary<string, long> { { "none", 50 } } }
            };
            SearchDescriptor<AuditLogElasticEntry> searchDescriptor = null;
            ConfigureMockTermFilterResponse(mockTermFilters).Callback((ISearchRequest request, CancellationToken token) => { searchDescriptor = request as SearchDescriptor<AuditLogElasticEntry>; });

            // Act
            var result = await auditLogSearchManager.GetTermFiltersAsync(null);

            // Assert
            result.Should().BeEquivalentTo(mockTermFilters);
            _mockElasticClient.Verify(_searchExpression, Times.Once);

            // Assert Correct Search Descriptor Behavior
            var searchToken = JToken.Parse(new ElasticClient().Serializer.SerializeToString(searchDescriptor));
            searchToken.Value<int>("size").Should().Be(0);

            // Assert Correct Aggregations
            var aggregationToken = searchToken.SelectToken("$.aggs", true);
            aggregationToken.SelectToken(".Name.terms").Value<string>("field").Should().BeEquivalentTo("name");
            aggregationToken.SelectToken(".Email.terms").Value<string>("field").Should().BeEquivalentTo("email");
            aggregationToken.SelectToken(".AmazonUID.terms").Value<string>("field").Should().BeEquivalentTo("amazonuid");
            aggregationToken.SelectToken(".Service.terms").Value<string>("field").Should().BeEquivalentTo("service");
            aggregationToken.SelectToken(".Handler.terms").Value<string>("field").Should().BeEquivalentTo("handler.keyword");
            aggregationToken.SelectToken(".Uri.terms").Value<string>("field").Should().BeEquivalentTo("uri.keyword");
            aggregationToken.SelectToken(".Method.terms").Value<string>("field").Should().BeEquivalentTo("method");
            aggregationToken.SelectToken(".ResponseCode.terms").Value<string>("field").Should().BeEquivalentTo("responsecode");
            aggregationToken.SelectToken(".Canceled.terms").Value<string>("field").Should().BeEquivalentTo("canceled");
            aggregationToken.SelectToken(".Exception.terms").Value<string>("field").Should().BeEquivalentTo("exception");
            aggregationToken.SelectToken(".ActionType.terms").Value<string>("field").Should().BeEquivalentTo("actiontype");
            aggregationToken.SelectToken(".OperationName.terms").Value<string>("field").Should().BeEquivalentTo("operationname");
        }

        // Note by Felix Kastner (fkastne):
        // This test is annoyingly large, but it is important to test that multiple filter conditions can be combined.
        [Fact]
        public async Task GetTermFiltersAsync_AllSearchOptionsGiven_ShouldConstructCorrectQuery()
        {
            // Arrange
            var unixTimeMilliseconds = DateTimeOffset.UtcNow.ToUnixTimeMilliseconds();
            var guid = Guid.NewGuid().ToString();
            var handler = "TestHandler";
            var auditLogSearchManager = CreateAuditLogSearchManager();
            var mockTermFilters = new Dictionary<string, Dictionary<string, long>>
            {
                { "Email", new Dictionary<string, long> { { "TestEmail1", 5 } } },
                { "Name", new Dictionary<string, long> { { "none", 50 } } }
            };
            SearchDescriptor<AuditLogElasticEntry> searchDescriptor = null;
            ConfigureMockTermFilterResponse(mockTermFilters).Callback((ISearchRequest request, CancellationToken token) => { searchDescriptor = request as SearchDescriptor<AuditLogElasticEntry>; });
            AuditLogSearch searchCriteria = new AuditLogSearch
            {
                SearchFilter = new ScrollingSearchFilter
                {
                    Direction = SearchDirection.Newer,
                    Timestamp = unixTimeMilliseconds,
                    Id = guid
                },
                Query = new AuditLogSearchQuery
                {
                    Handler = handler
                },
                Filter = new AuditLogSearchFilter
                {
                    ActionType = new[] { "read", "write" },
                    Name = new[] { "name1", "name2", "name3" }
                }
            };

            // Act
            var result = await auditLogSearchManager.GetTermFiltersAsync(searchCriteria);

            // Assert
            result.Should().BeEquivalentTo(mockTermFilters);
            _mockElasticClient.Verify(_searchExpression, Times.Once);

            // Assert Correct Search Descriptor Behavior
            var searchToken = JToken.Parse(new ElasticClient().Serializer.SerializeToString(searchDescriptor));
            searchToken.Value<int>("size").Should().Be(0);

            // Assert Correct Aggregations
            var aggregationToken = searchToken.SelectToken("$.aggs", true);
            aggregationToken.SelectToken(".Name.terms").Value<string>("field").Should().BeEquivalentTo("name");
            aggregationToken.SelectToken(".Email.terms").Value<string>("field").Should().BeEquivalentTo("email");
            aggregationToken.SelectToken(".AmazonUID.terms").Value<string>("field").Should().BeEquivalentTo("amazonuid");
            aggregationToken.SelectToken(".Service.terms").Value<string>("field").Should().BeEquivalentTo("service");
            aggregationToken.SelectToken(".Handler.terms").Value<string>("field").Should().BeEquivalentTo("handler.keyword");
            aggregationToken.SelectToken(".Uri.terms").Value<string>("field").Should().BeEquivalentTo("uri.keyword");
            aggregationToken.SelectToken(".Method.terms").Value<string>("field").Should().BeEquivalentTo("method");
            aggregationToken.SelectToken(".ResponseCode.terms").Value<string>("field").Should().BeEquivalentTo("responsecode");
            aggregationToken.SelectToken(".Canceled.terms").Value<string>("field").Should().BeEquivalentTo("canceled");
            aggregationToken.SelectToken(".Exception.terms").Value<string>("field").Should().BeEquivalentTo("exception");
            aggregationToken.SelectToken(".ActionType.terms").Value<string>("field").Should().BeEquivalentTo("actiontype");
            aggregationToken.SelectToken(".OperationName.terms").Value<string>("field").Should().BeEquivalentTo("operationname");

            // Assert that Search After was correctly applied:
            searchToken.SelectToken("$.sort..created").Value<string>("order").Should().Be("asc");
            searchToken.SelectToken("$.sort..id").Value<string>("order").Should().Be("desc");
            searchToken.SelectToken("search_after").Values<string>().Should().ContainInOrder(unixTimeMilliseconds.ToString(), guid);

            // Assert that Text Search was correctly applied:
            var handlerToken = searchToken.SelectToken("query.bool.must[0].match.handler", true);
            handlerToken.Value<string>("query").Should().Be(handler);

            // Assert that Term Filters were correctly applied:
            var filterToken = searchToken.SelectToken("$.query.bool.filter", true);
            filterToken.SelectToken("..terms.actionType").Values<string>().Should().BeEquivalentTo(searchCriteria.Filter.ActionType);
            filterToken.SelectToken("..terms.name").Values<string>().Should().BeEquivalentTo(searchCriteria.Filter.Name);
        }

        [Fact]
        public async Task GetTermFiltersAsync_InvalidResponseGiven_ShouldReturnEmptyDictionary()
        {
            // Arrange
            var mockTermFilters = new Dictionary<string, Dictionary<string, long>>
            {
                { "Source", new Dictionary<string, long> { { "TestName1", 5 } } },
                { "Username", new Dictionary<string, long> { { "none", 50 } } }
            };
            ConfigureMockTermFilterResponse(mockTermFilters, false);
            var auditLogSearchManager = CreateAuditLogSearchManager();

            // Act
            var result = await auditLogSearchManager.GetTermFiltersAsync(null);

            // Assert
            result.Should().BeEmpty();
        }
    }
}
