﻿using Moq;
using System;
using System.Threading;
using System.Threading.Tasks;
using Amazon.SQS.Model;
using FluentAssertions;
using Microsoft.Extensions.Logging.Abstractions;
using Newtonsoft.Json;
using Twitch.AuditLogService.ChangeLog;
using Twitch.AuditLogService.ChangeLog.Models;
using Twitch.AuditLogService.Models;
using Twitch.Shared.Sqs.Models;
using Xunit;

namespace Twitch.AuditLogService.Tests.ChangeLog
{
    [Trait("Category", "Unit")]
    public class ChangeLogMessageProcessorTests : IDisposable
    {
        private const string ChangeLogEntryGuid = "unique_changelog_entry_id";

        private readonly Message _sqsMessage = new Message { Body = "random message " };
        private readonly Message _snsSubscriptionConfirmationMessage = new Message { Body = @"{ ""Type"":""SubscriptionConfirmation"", ""SigningCertURL"":""https://sns"" } " };

        private readonly Message _snsAuditLogNotificationMessage = new Message
        {
            Body = JsonConvert.SerializeObject(
                new
                {
                    Type = "Notification",
                    SigningCertURL = "https://sns",
                    Message = JsonConvert.SerializeObject(new AuditLogElasticEntry { Created = 1562632940414 })
                }
            )
        };

        private readonly Message _snsChangeLogNotificationMessage = new Message
        {
            Body = JsonConvert.SerializeObject(
                new
                {
                    Type = "Notification",
                    SigningCertURL = "https://sns",
                    Message = JsonConvert.SerializeObject(new ChangeLogElasticEntry
                    {
                        Id = ChangeLogEntryGuid,
                        Timestamp = 1562632940
                    })
                }
            )
        };

        private readonly MockRepository _mockRepository;
        private readonly Mock<IChangeLogSearchManager> _mockChangeLogSearchManager;

        public ChangeLogMessageProcessorTests()
        {
            _mockRepository = new MockRepository(MockBehavior.Strict);
            _mockChangeLogSearchManager = _mockRepository.Create<IChangeLogSearchManager>();
        }

        public void Dispose()
        {
            _mockRepository.VerifyAll();
        }

        private ChangeLogMessageProcessor CreateChangeLogMessageProcessor()
        {
            return new ChangeLogMessageProcessor(
                new NullLogger<ChangeLogMessageProcessor>(), 
                _mockChangeLogSearchManager.Object);
        }

        [Fact]
        public async Task TryProcessMessagesAsync_NonSnsMessageGiven_ShouldNotProcessMessage()
        {
            // Arrange
            var changeLogMessageProcessor = CreateChangeLogMessageProcessor();
            var messages = new[] { WrappedSqsMessage.CreateFromMessage(_sqsMessage) };

            // Act
            var result = await changeLogMessageProcessor.TryProcessMessagesAsync(messages, CancellationToken.None);

            // Assert
            result.Should().BeEmpty();
        }

        [Fact]
        public async Task TryProcessMessagesAsync_SnsSubscriptionConfirmationMessageGiven_ShouldNotProcessMessage()
        {
            // Arrange
            var changeLogMessageProcessor = CreateChangeLogMessageProcessor();
            var messages = new[] { WrappedSqsMessage.CreateFromMessage(_snsSubscriptionConfirmationMessage) };

            // Act
            var result = await changeLogMessageProcessor.TryProcessMessagesAsync(messages, CancellationToken.None);

            // Assert
            result.Should().BeEmpty();
        }

        [Fact]
        public async Task TryProcessMessagesAsync_SnsAuditLogNotificationMessageGiven_ShouldNotProcessMessage()
        {
            // Arrange
            var changeLogMessageProcessor = CreateChangeLogMessageProcessor();
            var messages = new[] { WrappedSqsMessage.CreateFromMessage(_snsAuditLogNotificationMessage) };

            // Act
            var result = await changeLogMessageProcessor.TryProcessMessagesAsync(messages, CancellationToken.None);

            // Assert
            result.Should().BeEmpty();
        }

        [Fact]
        public async Task TryProcessMessagesAsync_SnsChangeLogNotificationMessageGiven_ShouldProcessMessage()
        {
            // Arrange
            var changeLogMessageProcessor = CreateChangeLogMessageProcessor();
            var messages = new[] { WrappedSqsMessage.CreateFromMessage(_snsChangeLogNotificationMessage) };
            _mockChangeLogSearchManager.Setup(sm => sm.IndexAsync(It.Is<ChangeLogElasticEntry>(entry => entry.Id.Equals(ChangeLogEntryGuid, StringComparison.Ordinal)))).ReturnsAsync(true);

            // Act
            var result = await changeLogMessageProcessor.TryProcessMessagesAsync(messages, CancellationToken.None);

            // Assert
            // ReSharper disable once CoVariantArrayConversion
            result.Should().BeEquivalentTo(messages);
        }

        [Fact]
        public async Task TryProcessMessagesAsync_IndexAsyncThrowsExceptionGiven_ShouldNotThrow()
        {
            // Arrange
            var changeLogMessageProcessor = CreateChangeLogMessageProcessor();
            var messages = new[] { WrappedSqsMessage.CreateFromMessage(_snsChangeLogNotificationMessage) };
            _mockChangeLogSearchManager.Setup(sm => sm.IndexAsync(It.IsAny<ChangeLogElasticEntry>())).Throws<InvalidOperationException>();

            // Act
            var result = await changeLogMessageProcessor.TryProcessMessagesAsync(messages, CancellationToken.None);

            // Assert
            result.Should().BeEmpty();
        }
    }
}
