﻿using Microsoft.Extensions.Caching.Memory;
using Microsoft.Extensions.Options;
using Moq;
using Nest;
using System;
using System.Collections.Generic;
using System.Linq.Expressions;
using System.Threading;
using System.Threading.Tasks;
using Elasticsearch.Net;
using FluentAssertions;
using Microsoft.Extensions.Logging.Abstractions;
using Moq.Language.Flow;
using Newtonsoft.Json.Linq;
using Twitch.AuditLogService.ChangeLog;
using Twitch.AuditLogService.ChangeLog.Models;
using Twitch.ElasticSearch;
using Twitch.ElasticSearch.Search;
using Xunit;

namespace Twitch.AuditLogService.Tests.ChangeLog
{
    [Trait("Category", "Unit")]
    public class ChangeLogSearchManagerTests : IDisposable
    {
        private readonly MockRepository _mockRepository;

        private readonly Mock<IElasticClient> _mockElasticClient;
        private readonly Mock<IOptions<SearchConfiguration>> _mockOptions;
        private readonly Mock<IIndexResponse> _indexResponseMock;

        public ChangeLogSearchManagerTests()
        {
            _mockRepository = new MockRepository(MockBehavior.Strict);

            _mockElasticClient = _mockRepository.Create<IElasticClient>();
            _mockOptions = _mockRepository.Create<IOptions<SearchConfiguration>>();
            _mockOptions.SetupGet(o => o.Value).Returns(new SearchConfiguration { TimeSeriesType = TimeSeriesIndexType.Monthly });

            var apiCallDetailsMock = _mockRepository.Create<IApiCallDetails>();
            apiCallDetailsMock.SetupGet(e => e.Success).Returns(true);

            _indexResponseMock = _mockRepository.Create<IIndexResponse>();

            var existsResponseMock = _mockRepository.Create<IExistsResponse>();
            existsResponseMock.SetupGet(e => e.Exists).Returns(true);
            existsResponseMock.SetupGet(e => e.ApiCall).Returns(apiCallDetailsMock.Object);

            _mockElasticClient.Setup(m => m.IndexTemplateExists(It.IsAny<Name>(), null)).Returns(existsResponseMock.Object);
        }

        public void Dispose()
        {
            _mockRepository.VerifyAll();
        }

        private ChangeLogSearchManager CreateChangeLogSearchManager()
        {
            return new ChangeLogSearchManager(
                _mockElasticClient.Object,
                _mockOptions.Object,
                new NullLogger<ChangeLogSearchManager>(),
                new MemoryCache(new MemoryCacheOptions()));
        }

        private readonly Expression<Func<IElasticClient, Task<ISearchResponse<ChangeLogElasticEntry>>>> _searchExpression = ec =>
            ec.SearchAsync<ChangeLogElasticEntry>(It.IsAny<ISearchRequest>(), It.IsAny<CancellationToken>());
        private IReturnsResult<IElasticClient> ConfigureMockSearchResponse(ChangeLogElasticEntry[] mockEntries, bool validResponse = true)
        {
            var searchResponseMock = new Mock<ISearchResponse<ChangeLogElasticEntry>>();
            searchResponseMock.SetupGet(x => x.IsValid).Returns(validResponse);
            searchResponseMock.Setup(x => x.Documents).Returns(mockEntries);
            return _mockElasticClient.Setup(_searchExpression).ReturnsAsync(searchResponseMock.Object);
        }
        
        private IReturnsResult<IElasticClient> ConfigureMockTermFilterResponse(Dictionary<string, Dictionary<string, long>> mockTermFilters, bool validResponse = true)
        {
            var results = new Dictionary<string, IAggregate>();
            foreach (var mockTermFilter in mockTermFilters)
            {
                var innerAggs = new List<IBucket>();
                foreach (var term in mockTermFilter.Value)
                {
                    innerAggs.Add(new KeyedBucket<object> { Key = term.Key, DocCount = term.Value });
                }

                var agg = new BucketAggregate
                {
                    DocCount = mockTermFilter.Value.Count,
                    Items = innerAggs
                };
                results[mockTermFilter.Key] = agg;
            }
            var searchResponseMock = new Mock<ISearchResponse<ChangeLogElasticEntry>>();
            searchResponseMock.SetupGet(x => x.IsValid).Returns(validResponse);
            searchResponseMock.SetupGet(x => x.Aggregations).Returns(results);
            searchResponseMock.SetupGet(x => x.Aggs).Returns(new AggregationsHelper(results));
            return _mockElasticClient.Setup(_searchExpression).ReturnsAsync(searchResponseMock.Object);
        }

        [Fact]
        public async Task IndexAsync_ValidItemGiven_ShouldIndexItem()
        {
            // Arrange
            var item = new ChangeLogElasticEntry { Timestamp = DateTimeOffset.UtcNow.ToUnixTimeMilliseconds(), Id = Guid.NewGuid().ToString() };
            Expression<Func<IElasticClient, Task<IIndexResponse>>> indexExpression = ec => 
                ec.IndexAsync(It.Is<IndexRequest<ChangeLogElasticEntry>>(ir => ir.Document == item), null, It.IsAny<CancellationToken>());
            _indexResponseMock.SetupGet(ir => ir.IsValid).Returns(true);
            _mockElasticClient.Setup(indexExpression).ReturnsAsync(_indexResponseMock.Object);
            var changeLogSearchManager = CreateChangeLogSearchManager();
            
            // Act
            var result = await changeLogSearchManager.IndexAsync(item);

            // Assert
            result.Should().BeTrue();
            _mockElasticClient.Verify(indexExpression, Times.Once);
        }

        [Fact]
        public async Task IndexAsync_NullItemGiven_ShouldThrowArgumentNullException()
        {
            // Arrange
            var changeLogSearchManager = CreateChangeLogSearchManager();

            // Act
            Func<Task<bool>> action = async () => await changeLogSearchManager.IndexAsync(null);

            // Assert
            await action.Should().ThrowAsync<ArgumentNullException>();
        }

        [Fact]
        public async Task IndexAsync_InvalidResponseGiven_ShouldReturnFalse()
        {
            // Arrange
            var item = new ChangeLogElasticEntry { Timestamp = DateTimeOffset.UtcNow.ToUnixTimeMilliseconds(), Id = Guid.NewGuid().ToString() };
            _indexResponseMock.SetupGet(ir => ir.IsValid).Returns(false);
            _indexResponseMock.SetupGet(ir => ir.OriginalException).Returns(new Exception());
            _indexResponseMock.SetupGet(ir => ir.DebugInformation).Returns("test failure");
            _mockElasticClient.Setup(m => m.IndexAsync(It.Is<IndexRequest<ChangeLogElasticEntry>>(ir => ir.Document == item), null, It.IsAny<CancellationToken>()))
                .ReturnsAsync(_indexResponseMock.Object);
            var changeLogSearchManager = CreateChangeLogSearchManager();
            
            // Act
            var result = await changeLogSearchManager.IndexAsync(item);

            // Assert
            result.Should().BeFalse();
        }

        [Fact]
        public async Task SearchAsync_ScrollingSearchFilterGiven_ShouldConstructCorrectQuery()
        {
            // Arrange
            var unixTimeMilliseconds = DateTimeOffset.UtcNow.ToUnixTimeMilliseconds();
            var guid = Guid.NewGuid().ToString();
            var mockEntries = new[] { new ChangeLogElasticEntry { Timestamp = unixTimeMilliseconds, Id = guid } };
            SearchDescriptor<ChangeLogElasticEntry> searchDescriptor = null;
            ConfigureMockSearchResponse(mockEntries).Callback((ISearchRequest request, CancellationToken token) => { searchDescriptor = request as SearchDescriptor<ChangeLogElasticEntry>; });
            
            var changeLogSearchManager = CreateChangeLogSearchManager();
            ChangeLogSearch searchCriteria = new ChangeLogSearch 
            { 
                SearchFilter = new ScrollingSearchFilter
                {
                    Direction = SearchDirection.Newer,
                    Timestamp = unixTimeMilliseconds,
                    Id = guid
                }
            };
            int pageSize = 500;

            // Act
            var result = await changeLogSearchManager.SearchAsync(searchCriteria, pageSize);

            // Assert
            // ReSharper disable once CoVariantArrayConversion
            result.Should().BeEquivalentTo(mockEntries);
            _mockElasticClient.Verify(_searchExpression, Times.Once);

            // Assert Correct Search Descriptor Behavior
            var searchToken = JToken.Parse(new ElasticClient().Serializer.SerializeToString(searchDescriptor));
            searchToken.Value<int>("size").Should().Be(pageSize);
            searchToken.SelectToken("$.sort..timestamp").Value<string>("order").Should().Be("asc");
            searchToken.SelectToken("$.sort..id").Value<string>("order").Should().Be("desc");
            searchToken.SelectToken("search_after").Values<string>().Should().ContainInOrder(unixTimeMilliseconds.ToString(), guid);
        }

        [Fact]
        public async Task SearchAsync_SearchQueryGiven_ShouldConstructCorrectQuery()
        {
            // Arrange
            var username = "TestUser";
            var mockEntries = new[] { new ChangeLogElasticEntry { Timestamp = DateTimeOffset.UtcNow.ToUnixTimeMilliseconds(), Id = Guid.NewGuid().ToString() } };
            SearchDescriptor<ChangeLogElasticEntry> searchDescriptor = null;
            ConfigureMockSearchResponse(mockEntries).Callback((ISearchRequest request, CancellationToken token) => { searchDescriptor = request as SearchDescriptor<ChangeLogElasticEntry>; });

            var changeLogSearchManager = CreateChangeLogSearchManager();
            ChangeLogSearch searchCriteria = new ChangeLogSearch
            {
                Query = new ChangeLogSearchQuery
                {
                    Username = username
                }
            };
            int pageSize = 500;

            // Act
            var result = await changeLogSearchManager.SearchAsync(searchCriteria, pageSize);

            // Assert
            // ReSharper disable once CoVariantArrayConversion
            result.Should().BeEquivalentTo(mockEntries);
            _mockElasticClient.Verify(_searchExpression, Times.Once);

            // Assert Correct Search Descriptor Behavior
            var searchToken = JToken.Parse(new ElasticClient().Serializer.SerializeToString(searchDescriptor));
            var usernameToken = searchToken.SelectToken("query.match.username", true);
            usernameToken.Value<string>("query").Should().Be(username);
        }

        [Fact]
        public async Task SearchAsync_SearchFilterGiven_ShouldConstructCorrectQuery()
        {
            // Arrange
            var mockEntries = new[] { new ChangeLogElasticEntry { Timestamp = DateTimeOffset.UtcNow.ToUnixTimeMilliseconds(), Id = Guid.NewGuid().ToString() } };
            SearchDescriptor<ChangeLogElasticEntry> searchDescriptor = null;
            ConfigureMockSearchResponse(mockEntries).Callback((ISearchRequest request, CancellationToken token) => { searchDescriptor = request as SearchDescriptor<ChangeLogElasticEntry>; });

            var changeLogSearchManager = CreateChangeLogSearchManager();
            ChangeLogSearch searchCriteria = new ChangeLogSearch
            {
                Filter = new ChangeLogSearchFilter
                {
                    Categories = new[] { "cat1", "cat2" },
                    Criticality = new [] {"crit1", "crit2" },
                    Source = new [] { "source1", "source2", "source3" },
                    Username = new []{ "TestUser" }
                }
            };
            int pageSize = 500;

            // Act
            var result = await changeLogSearchManager.SearchAsync(searchCriteria, pageSize);

            // Assert
            // ReSharper disable once CoVariantArrayConversion
            result.Should().BeEquivalentTo(mockEntries);
            _mockElasticClient.Verify(_searchExpression, Times.Once);

            // Assert Correct Search Descriptor Behavior
            var searchToken = JToken.Parse(new ElasticClient().Serializer.SerializeToString(searchDescriptor));
            var filterToken = searchToken.SelectToken("$.query.bool.filter", true);
            filterToken.SelectToken("..terms.categories").Values<string>().Should().BeEquivalentTo(searchCriteria.Filter.Categories);
            filterToken.SelectToken("..terms.criticality").Values<string>().Should().BeEquivalentTo(searchCriteria.Filter.Criticality);
            filterToken.SelectToken("..terms.['source.keyword']").Values<string>().Should().BeEquivalentTo(searchCriteria.Filter.Source);
            filterToken.SelectToken("..terms.['username.keyword']").Values<string>().Should().BeEquivalentTo(searchCriteria.Filter.Username);
        }

        // Note by Felix Kastner (fkastne):
        // This test is annoyingly large, but it is important to test that multiple filter conditions can be combined.
        [Fact]
        public async Task SearchAsync_AllSearchOptionsGiven_ShouldConstructCorrectQuery()
        {
            // Arrange
            var unixTimeMilliseconds = DateTimeOffset.UtcNow.ToUnixTimeMilliseconds();
            var guid = Guid.NewGuid().ToString();
            var username = "TestUser";
            var mockEntries = new[] { new ChangeLogElasticEntry { Timestamp = unixTimeMilliseconds, Id = guid } };
            SearchDescriptor<ChangeLogElasticEntry> searchDescriptor = null;
            ConfigureMockSearchResponse(mockEntries).Callback((ISearchRequest request, CancellationToken token) => { searchDescriptor = request as SearchDescriptor<ChangeLogElasticEntry>; });

            var changeLogSearchManager = CreateChangeLogSearchManager();
            ChangeLogSearch searchCriteria = new ChangeLogSearch
            {
                SearchFilter = new ScrollingSearchFilter
                {
                    Direction = SearchDirection.Newer,
                    Timestamp = unixTimeMilliseconds,
                    Id = guid
                },
                Query = new ChangeLogSearchQuery
                {
                    Username = username
                },
                Filter = new ChangeLogSearchFilter
                {
                    Categories = new[] { "cat1", "cat2" },
                    Criticality = new[] { "crit1", "crit2" },
                    Source = new[] { "source1", "source2", "source3" },
                    Username = new[] { "TestUser" }
                }
            };
            int pageSize = 500;

            // Act
            var result = await changeLogSearchManager.SearchAsync(searchCriteria, pageSize);

            // Assert
            // ReSharper disable once CoVariantArrayConversion
            result.Should().BeEquivalentTo(mockEntries);
            _mockElasticClient.Verify(_searchExpression, Times.Once);

            // Assert Correct Search Descriptor Behavior
            var searchToken = JToken.Parse(new ElasticClient().Serializer.SerializeToString(searchDescriptor));
            searchToken.Value<int>("size").Should().Be(pageSize);

            // Assert that Search After was correctly applied:
            searchToken.SelectToken("$.sort..timestamp").Value<string>("order").Should().Be("asc");
            searchToken.SelectToken("$.sort..id").Value<string>("order").Should().Be("desc");
            searchToken.SelectToken("search_after").Values<string>().Should().ContainInOrder(unixTimeMilliseconds.ToString(), guid);

            // Assert that Text Search was correctly applied:
            var usernameToken = searchToken.SelectToken("query.bool.must[0].match.username", true);
            usernameToken.Value<string>("query").Should().Be(username);

            // Assert that Term Filters were correctly applied:
            var filterToken = searchToken.SelectToken("$.query.bool.filter", true);
            filterToken.SelectToken("..terms.categories").Values<string>().Should().BeEquivalentTo(searchCriteria.Filter.Categories);
            filterToken.SelectToken("..terms.criticality").Values<string>().Should().BeEquivalentTo(searchCriteria.Filter.Criticality);
            filterToken.SelectToken("..terms.['source.keyword']").Values<string>().Should().BeEquivalentTo(searchCriteria.Filter.Source);
            filterToken.SelectToken("..terms.['username.keyword']").Values<string>().Should().BeEquivalentTo(searchCriteria.Filter.Username);
        }

        [Fact]
        public async Task SearchAsync_InvalidResponseGiven_ShouldReturnEmptyCollection()
        {
            // Arrange
            var mockEntries = new[] { new ChangeLogElasticEntry { Timestamp = DateTimeOffset.UtcNow.ToUnixTimeMilliseconds(), Id = Guid.NewGuid().ToString() } };
            ConfigureMockSearchResponse(mockEntries, false);
            var changeLogSearchManager = CreateChangeLogSearchManager();

            // Act
            var result = await changeLogSearchManager.SearchAsync(new ChangeLogSearch(), 500);

            // Assert
            result.Should().BeEmpty();
        }

        [Fact]
        public async Task GetTermFiltersAsync_ShouldConstructCorrectQuery()
        {
            // Arrange
            var changeLogSearchManager = CreateChangeLogSearchManager();
            var mockTermFilters = new Dictionary<string, Dictionary<string, long>>
            {
                { "Source", new Dictionary<string, long> { { "TestName1", 5 } } },
                { "Username", new Dictionary<string, long> { { "none", 50 } } }
            };
            SearchDescriptor<ChangeLogElasticEntry> searchDescriptor = null;
            ConfigureMockTermFilterResponse(mockTermFilters).Callback((ISearchRequest request, CancellationToken token) => { searchDescriptor = request as SearchDescriptor<ChangeLogElasticEntry>; });

            // Act
            var result = await changeLogSearchManager.GetTermFiltersAsync(null);

            // Assert
            result.Should().BeEquivalentTo(mockTermFilters);
            _mockElasticClient.Verify(_searchExpression, Times.Once);

            // Assert Correct Search Descriptor Behavior
            var searchToken = JToken.Parse(new ElasticClient().Serializer.SerializeToString(searchDescriptor));
            searchToken.Value<int>("size").Should().Be(0);

            // Assert Correct Aggregations
            var aggregationToken = searchToken.SelectToken("$.aggs", true);
            aggregationToken.SelectToken(".Username.terms").Value<string>("field").Should().BeEquivalentTo("username.keyword");
            aggregationToken.SelectToken(".Source.terms").Value<string>("field").Should().BeEquivalentTo("source.keyword");
            aggregationToken.SelectToken(".Categories.terms").Value<string>("field").Should().BeEquivalentTo("categories");
            aggregationToken.SelectToken(".Criticality.terms").Value<string>("field").Should().BeEquivalentTo("criticality");
        }

        // Note by Felix Kastner (fkastne):
        // This test is annoyingly large, but it is important to test that multiple filter conditions can be combined.
        [Fact]
        public async Task GetTermFiltersAsync_AllSearchOptionsGiven_ShouldConstructCorrectQuery()
        {
            // Arrange
            var unixTimeMilliseconds = DateTimeOffset.UtcNow.ToUnixTimeMilliseconds();
            var guid = Guid.NewGuid().ToString();
            var username = "TestUser";
            var changeLogSearchManager = CreateChangeLogSearchManager();
            var mockTermFilters = new Dictionary<string, Dictionary<string, long>>
            {
                { "Source", new Dictionary<string, long> { { "TestName1", 5 } } },
                { "Username", new Dictionary<string, long> { { "none", 50 } } }
            };
            SearchDescriptor<ChangeLogElasticEntry> searchDescriptor = null;
            ConfigureMockTermFilterResponse(mockTermFilters).Callback((ISearchRequest request, CancellationToken token) => { searchDescriptor = request as SearchDescriptor<ChangeLogElasticEntry>; });
            ChangeLogSearch searchCriteria = new ChangeLogSearch
            {
                SearchFilter = new ScrollingSearchFilter
                {
                    Direction = SearchDirection.Newer,
                    Timestamp = unixTimeMilliseconds,
                    Id = guid
                },
                Query = new ChangeLogSearchQuery
                {
                    Username = username
                },
                Filter = new ChangeLogSearchFilter
                {
                    Categories = new[] { "cat1", "cat2" },
                    Criticality = new[] { "crit1", "crit2" },
                    Source = new[] { "source1", "source2", "source3" },
                    Username = new[] { "TestUser" }
                }
            };

            // Act
            var result = await changeLogSearchManager.GetTermFiltersAsync(searchCriteria);

            // Assert
            result.Should().BeEquivalentTo(mockTermFilters);
            _mockElasticClient.Verify(_searchExpression, Times.Once);

            // Assert Correct Search Descriptor Behavior
            var searchToken = JToken.Parse(new ElasticClient().Serializer.SerializeToString(searchDescriptor));
            searchToken.Value<int>("size").Should().Be(0);

            // Assert Correct Aggregations
            var aggregationToken = searchToken.SelectToken("$.aggs", true);
            aggregationToken.SelectToken(".Username.terms").Value<string>("field").Should().BeEquivalentTo("username.keyword");
            aggregationToken.SelectToken(".Source.terms").Value<string>("field").Should().BeEquivalentTo("source.keyword");
            aggregationToken.SelectToken(".Categories.terms").Value<string>("field").Should().BeEquivalentTo("categories");
            aggregationToken.SelectToken(".Criticality.terms").Value<string>("field").Should().BeEquivalentTo("criticality");

            // Assert that Search After was correctly applied:
            searchToken.SelectToken("$.sort..timestamp").Value<string>("order").Should().Be("asc");
            searchToken.SelectToken("$.sort..id").Value<string>("order").Should().Be("desc");
            searchToken.SelectToken("search_after").Values<string>().Should().ContainInOrder(unixTimeMilliseconds.ToString(), guid);

            // Assert that Text Search was correctly applied:
            var usernameToken = searchToken.SelectToken("query.bool.must[0].match.username", true);
            usernameToken.Value<string>("query").Should().Be(username);

            // Assert that Term Filters were correctly applied:
            var filterToken = searchToken.SelectToken("$.query.bool.filter", true);
            filterToken.SelectToken("..terms.categories").Values<string>().Should().BeEquivalentTo(searchCriteria.Filter.Categories);
            filterToken.SelectToken("..terms.criticality").Values<string>().Should().BeEquivalentTo(searchCriteria.Filter.Criticality);
            filterToken.SelectToken("..terms.['source.keyword']").Values<string>().Should().BeEquivalentTo(searchCriteria.Filter.Source);
            filterToken.SelectToken("..terms.['username.keyword']").Values<string>().Should().BeEquivalentTo(searchCriteria.Filter.Username);
        }

        [Fact]
        public async Task GetTermFiltersAsync_InvalidResponseGiven_ShouldReturnEmptyDictionary()
        {
            // Arrange
            var mockTermFilters = new Dictionary<string, Dictionary<string, long>>
            {
                { "Source", new Dictionary<string, long> { { "TestName1", 5 } } },
                { "Username", new Dictionary<string, long> { { "none", 50 } } }
            };
            ConfigureMockTermFilterResponse(mockTermFilters, false);
            var changeLogSearchManager = CreateChangeLogSearchManager();

            // Act
            var result = await changeLogSearchManager.GetTermFiltersAsync(null);

            // Assert
            result.Should().BeEmpty();
        }
    }
}
