﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Text;
using System.Threading.Tasks;
using FluentAssertions;
using Newtonsoft.Json;
using Twitch.AuditLogService.ChangeLog.Models;
using Twitch.AuditLogService.Models;
using Twitch.AuditLogService.Web;
using Twitch.AuditLogService.Web.Model;
using Twitch.Shared.Test;
using Xunit;

namespace Twitch.AuditLogService.Tests.Integration.Controllers
{
    [Trait("Category", "Integration")]
    public class SearchControllerTests : ControllerTestBase<Startup>
    {
        public SearchControllerTests(LocalWebApplicationFixture fixture) : base(fixture)
        {
        }

        [Fact]
        public async Task Search_ValidSearchCriteriaGiven_ShouldReturnMatch()
        {
            // Arrange
            var request = new AuditLogSearch();

            // Act
            var response = await Client.PostAsJsonAsync("/search", request);
            
            // Assert
            response.StatusCode.Should().Be(HttpStatusCode.OK);
            var matches = JsonConvert.DeserializeObject<IEnumerable<AuditLogElasticEntry>>(await response.Content.ReadAsStringAsync()).ToList();
            matches.Should().HaveCount(1);
            matches.Should().Contain(ee => ee.Name == "TestName1");
        }

        [Fact]
        public async Task Search_NullSearchCriteriaGiven_ShouldReturnBadRequest()
        {
            // Arrange
            var content = new StringContent(string.Empty, Encoding.UTF8, "application/json");

            // Act
            var response = await Client.PostAsync("/search", content);

            // Assert
            response.StatusCode.Should().Be(HttpStatusCode.BadRequest);
        }

        [Fact]
        public async Task GetSearchOptions_ShouldReturnOptions()
        {
            // Act
            var response = await Client.GetAsync("/searchoptions");

            // Assert
            response.StatusCode.Should().Be(HttpStatusCode.OK);
            var searchOptions = JsonConvert.DeserializeObject<SearchOptions>(await response.Content.ReadAsStringAsync());
            searchOptions.Filter.Should().ContainKey("Name")
                .WhichValue
                .Should().ContainKey("TestName1").WhichValue.Should().Be(5);
            searchOptions.Query.Should().Contain("Manager");
        }

        [Fact]
        public async Task GetSearchOptions_ValidSearchCriteriaGiven_ShouldReturnOptions()
        {
            // Arrange
            var request = new AuditLogSearch();

            // Act
            var response = await Client.PostAsJsonAsync("/searchoptions", request);

            // Assert
            response.StatusCode.Should().Be(HttpStatusCode.OK);
            var searchOptions = JsonConvert.DeserializeObject<SearchOptions>(await response.Content.ReadAsStringAsync());
            searchOptions.Filter.Should().ContainKey("Name")
                .WhichValue
                .Should().ContainKey("TestName1").WhichValue.Should().Be(5);
            searchOptions.Query.Should().Contain("Manager");
        }

        [Fact]
        public async Task ChangeLogSearch_ValidSearchCriteriaGiven_ShouldReturnMatch()
        {
            // Arrange
            var request = new ChangeLogSearch();

            // Act
            var response = await Client.PostAsJsonAsync("/changelog/search", request);
            
            // Assert
            response.StatusCode.Should().Be(HttpStatusCode.OK);
            var matches = JsonConvert.DeserializeObject<IEnumerable<ChangeLogElasticEntry>>(await response.Content.ReadAsStringAsync()).ToList();
            matches.Should().HaveCount(1);
            matches.Should().Contain(ee => ee.Username == "TestName1");
        }

        [Fact]
        public async Task ChangeLogSearch_NullSearchCriteriaGiven_ShouldReturnBadRequest()
        {
            // Arrange
            var content = new StringContent(string.Empty, Encoding.UTF8, "application/json");

            // Act
            var response = await Client.PostAsync("/changelog/search", content);

            // Assert
            response.StatusCode.Should().Be(HttpStatusCode.BadRequest);
        }

        [Fact]
        public async Task GetChangeLogSearchOptions_ShouldReturnOptions()
        {
            // Act
            var response = await Client.GetAsync("/changelog/searchoptions");

            // Assert
            response.StatusCode.Should().Be(HttpStatusCode.OK);
            var searchOptions = JsonConvert.DeserializeObject<SearchOptions>(await response.Content.ReadAsStringAsync());
            searchOptions.Filter.Should().ContainKey("Source")
                .WhichValue
                .Should().ContainKey("TestName1").WhichValue.Should().Be(5);
            searchOptions.Query.Should().Contain("Command");
        }

        [Fact]
        public async Task GetChangeLogSearchOptions_ValidSearchCriteriaGiven_ShouldReturnOptions()
        {
            // Arrange
            var request = new ChangeLogSearch();

            // Act
            var response = await Client.PostAsJsonAsync("/changelog/searchoptions", request);

            // Assert
            response.StatusCode.Should().Be(HttpStatusCode.OK);
            var searchOptions = JsonConvert.DeserializeObject<SearchOptions>(await response.Content.ReadAsStringAsync());
            searchOptions.Filter.Should().ContainKey("Source")
                .WhichValue
                .Should().ContainKey("TestName1").WhichValue.Should().Be(5);
            searchOptions.Query.Should().Contain("Command");
        }

        [Fact]
        public async Task ChangeLogLegacySearch_ValidSearchOptionsGiven_ShouldReturnMatch()
        {
            // Arrange
            var currentSeconds = DateTimeOffset.UtcNow.ToUnixTimeSeconds();

            // Act
            var response = await Client.GetAsync($"/api/events?hours_from=5&until={currentSeconds}&criticality=4,5&category=null");

            // Assert
            response.StatusCode.Should().Be(HttpStatusCode.OK);
            var matches = JsonConvert.DeserializeObject<IEnumerable<ChangeLogElasticEntry>>(await response.Content.ReadAsStringAsync()).ToList();
            matches.Should().HaveCount(1);
            matches.Should().Contain(ee => ee.Username == "TestName1");
        }
    }
}
