﻿using System.Collections.Generic;
using System.Net;
using Microsoft.AspNetCore.Authentication.JwtBearer;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Authorization.Infrastructure;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.TestHost;
using Microsoft.Extensions.DependencyInjection;
using Microsoft.Extensions.DependencyInjection.Extensions;
using Microsoft.Extensions.Logging;
using Microsoft.Extensions.Logging.Abstractions;
using Moq;
using Twitch.AuditLogService.ChangeLog;
using Twitch.AuditLogService.ChangeLog.Models;
using Twitch.AuditLogService.Models;
using Twitch.AuditLogService.Web;
using Twitch.Shared.Test;
using Xunit;
using Xunit.Abstractions;
using Xunit.Sdk;

namespace Twitch.AuditLogService.Tests.Integration
{
    [CollectionDefinition(WebApplicationFixture.DefinitionName)]
    public class WebApplicationCollection : ICollectionFixture<LocalWebApplicationFixture> { }

    public class LocalWebApplicationFixture : WebApplicationFixture<Startup>
    {
        public LocalWebApplicationFixture(IMessageSink diagnosticMessageSink) : base(diagnosticMessageSink)
        {
            _mockRepository = new MockRepository(MockBehavior.Strict);
        }

        /// <summary>
        /// Sets the environment to Development.
        /// </summary>
        /// <param name="builder">The builder setting up the webhost.</param>
        protected override void ConfigureWebHost(IWebHostBuilder builder)
        {
            DiagnosticMessageSink.OnMessage(new DiagnosticMessage("Ran ConfigureWebHost"));

            builder.UseEnvironment("Development");

            builder.ConfigureTestServices(ConfigureTestServices);
        }


        private readonly MockRepository _mockRepository;
        
        private void ConfigureTestServices(IServiceCollection services)
        {
            services.Replace(ServiceDescriptor.Singleton(typeof(ILogger<>), typeof(NullLogger<>)));

            var mockAuditLogSearchManager = _mockRepository.Create<IAuditLogSearchManager>();
            var mockChangeLogSearchManager = _mockRepository.Create<IChangeLogSearchManager>();

            mockAuditLogSearchManager.Setup(sm => sm.GetTermFiltersAsync(It.IsAny<AuditLogSearch>()))
                .ReturnsAsync(new Dictionary<string, Dictionary<string, long>> { { "Name", new Dictionary<string, long> { { "TestName1", 5 } } } });
            mockAuditLogSearchManager.Setup(sm => sm.GetTermFiltersAsync(null))
                .ReturnsAsync(new Dictionary<string, Dictionary<string, long>> { { "Name", new Dictionary<string, long> { { "TestName1", 5 } } } });

            mockAuditLogSearchManager.Setup(
                sm => sm.SearchAsync(It.IsAny<AuditLogSearch>(), It.IsAny<int>())
            ).ReturnsAsync(new[] { new AuditLogElasticEntry { Name = "TestName1" } });

            services.Replace(ServiceDescriptor.Singleton(typeof(IAuditLogSearchManager), mockAuditLogSearchManager.Object));

            mockChangeLogSearchManager.Setup(sm => sm.GetTermFiltersAsync(It.IsAny<ChangeLogSearch>()))
                .ReturnsAsync(new Dictionary<string, Dictionary<string, long>> { { "Source", new Dictionary<string, long> { {"TestName1", 5 }}}});
            mockChangeLogSearchManager.Setup(sm => sm.GetTermFiltersAsync(null))
                .ReturnsAsync(new Dictionary<string, Dictionary<string, long>> { { "Source", new Dictionary<string, long> { {"TestName1", 5 }}}});

            mockChangeLogSearchManager.Setup(
                sm => sm.SearchAsync(It.IsAny<ChangeLogSearch>(), It.IsAny<int>())
            ).ReturnsAsync(new[] { new ChangeLogElasticEntry { Username = "TestName1" } });

            services.Replace(ServiceDescriptor.Singleton(typeof(IChangeLogSearchManager), mockChangeLogSearchManager.Object));

             // Set a default authorization policy that always returns success, effectively disabling authorization requirements.
            services.AddAuthorization(options =>
            {
                options.DefaultPolicy = new AuthorizationPolicy(
                    new[] {new AssertionRequirement(context => true)}, new[] {JwtBearerDefaults.AuthenticationScheme}
                );
            });
        }
    }
}