﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Threading.Tasks;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using Twitch.AuditLogClient.Util;
using Twitch.AuditLogService.ChangeLog;
using Twitch.AuditLogService.ChangeLog.Models;
using Twitch.AuditLogService.Models;
using Twitch.AuditLogService.Web.Model;
using Twitch.ElasticSearch.Search;
using Twitch.Shared.Extensions;
using Twitch.Shared.Util;

namespace Twitch.AuditLogService.Web.Controllers
{
    [Authorize]
    [ApiController]
    public class SearchController : ControllerBase
    {
        private static string[] _auditLogQueryParams;
        private static string[] _changeLogQueryParams;

        private readonly IAuditLogSearchManager _auditLogSearchManager;
        private readonly IChangeLogSearchManager _changeLogSearchManager;

        public SearchController(IAuditLogSearchManager auditLogSearchManager, IChangeLogSearchManager changeLogSearchManager)
        {
            _auditLogSearchManager = auditLogSearchManager;
            _changeLogSearchManager = changeLogSearchManager;
        }
        
        [HttpPost("search")]
        [RecordToAuditLog(AuditLogActionType.Read)]
        [ProducesResponseType(typeof(IReadOnlyCollection<AuditLogElasticEntry>), 200)]
        [ProducesResponseType(400)]
        public async Task<IActionResult> Search([FromBody]AuditLogSearch searchCriteria, [FromQuery]int pageSize = 500)
        {
            if (searchCriteria == null)
            {
                return BadRequest();
            }

            var searchResults = await _auditLogSearchManager.SearchAsync(searchCriteria, pageSize);
            return Ok(searchResults.OrderByDescending(sr => sr.Created));
        }

        /// <summary>
        /// Gets filterable terms and queryable fields that can be used as search options. 
        /// </summary>
        /// <returns></returns>
        [HttpGet("searchoptions")]
        [ProducesResponseType(typeof(SearchOptions), 200)]
        public async Task<IActionResult> GetSearchOptions()
        {
            InitializeAuditLogQueryParams();
            var filters = await _auditLogSearchManager.GetTermFiltersAsync(null);
            return Ok(new SearchOptions(filters, _auditLogQueryParams));
        }

        /// <summary>
        /// Gets filterable terms and queryable fields that can be used as search options, with the specified search options applied. 
        /// </summary>
        /// <returns></returns>
        [HttpPost("searchoptions", Name = "FilteredSearchOptions")]
        [RecordToAuditLog(AuditLogActionType.Read)]
        [ProducesResponseType(typeof(SearchOptions), 200)]
        public async Task<IActionResult> GetSearchOptions([FromBody]AuditLogSearch searchCriteria)
        {
            InitializeAuditLogQueryParams();
            var filters = await _auditLogSearchManager.GetTermFiltersAsync(searchCriteria);
            return Ok(new SearchOptions(filters, _auditLogQueryParams));
        }

        [HttpPost("changelog/search")]
        [RecordToAuditLog(AuditLogActionType.Read)]
        [ProducesResponseType(typeof(IReadOnlyCollection<ChangeLogElasticEntry>), 200)]
        [ProducesResponseType(400)]
        public async Task<IActionResult> ChangeLogSearch([FromBody]ChangeLogSearch searchCriteria, [FromQuery]int pageSize = 500)
        {
            if (searchCriteria == null)
            {
                return BadRequest();
            }

            var searchResults = await _changeLogSearchManager.SearchAsync(searchCriteria, pageSize);
            return Ok(searchResults.OrderByDescending(sr => sr.Timestamp));
        }

        /// <summary>
        /// Gets filterable terms and queryable fields that can be used as search options. 
        /// </summary>
        /// <returns></returns>
        [HttpGet("changelog/searchoptions")]
        [ProducesResponseType(typeof(SearchOptions), 200)]
        public async Task<IActionResult> GetChangeLogSearchOptions()
        {
            InitializeChangeLogQueryParams();
            var filters = await _changeLogSearchManager.GetTermFiltersAsync(null);
            return Ok(new SearchOptions(filters, _changeLogQueryParams));
        }

        /// <summary>
        /// Gets filterable terms and queryable fields that can be used as search options, with the specified search options applied. 
        /// </summary>
        /// <returns></returns>
        [HttpPost("changelog/searchoptions", Name = "ChangeLogFilteredSearchOptions")]
        [RecordToAuditLog(AuditLogActionType.Read)]
        [ProducesResponseType(typeof(SearchOptions), 200)]
        public async Task<IActionResult> GetChangeLogSearchOptions([FromBody]ChangeLogSearch searchCriteria)
        {
            InitializeChangeLogQueryParams();
            var filters = await _changeLogSearchManager.GetTermFiltersAsync(searchCriteria);
            return Ok(new SearchOptions(filters, _changeLogQueryParams));
        }

        [HttpGet("api/events")]
        [ProducesResponseType(typeof(IReadOnlyCollection<ChangeLogElasticEntry>), 200)]
        [ProducesResponseType(400)]
        public async Task<IActionResult> ChangeLogLegacySearch([FromQuery] ChangeLogLegacySearchOptions searchOptions)
        {
            Ensure.NotNull(nameof(searchOptions), searchOptions);
            var endDateTimeOffset = DateTimeOffset.UtcNow;
            if (searchOptions.Until > 0)
            {
                endDateTimeOffset = DateTimeOffset.FromUnixTimeSeconds(searchOptions.Until);
            }

            var searchCriteria = new ChangeLogSearch
            {
                SearchFilter = new ScrollingSearchFilter { Direction = SearchDirection.Newer, Timestamp = endDateTimeOffset.AddHours(-searchOptions.HoursAgo).ToUnixTimeMilliseconds() },
                Filter = new ChangeLogSearchFilter(),
                Query = new ChangeLogSearchQuery()
            };

            if (searchOptions.Criticality.HasValue())
            {
                var criticalities = searchOptions.Criticality.Split(',', StringSplitOptions.RemoveEmptyEntries);
                if (criticalities.Any())
                {
                    searchCriteria.Filter.Criticality = criticalities;
                }
            }

            if (searchOptions.Category.HasValue())
            {
                var categories = searchOptions.Category.Split(',', StringSplitOptions.RemoveEmptyEntries);
                if (categories.Any())
                {
                    searchCriteria.Filter.Categories = categories;
                }
            }

            searchCriteria.Query = new ChangeLogSearchQuery
            {
                Command = searchOptions.Command,
                Description = searchOptions.Description,
                AdditionalData = searchOptions.AdditionalData,
                Username = searchOptions.Username,
                Target = searchOptions.Target,
                Source = searchOptions.Source
            };

            var searchResults = await _changeLogSearchManager.SearchAsync(searchCriteria, 10_000);

            if (searchOptions.Until > 0)
            {
                searchResults = searchResults.Where(sr => sr.Timestamp <= endDateTimeOffset.ToUnixTimeMilliseconds()).ToList();
            }

            return Ok(searchResults.OrderByDescending(sr => sr.Timestamp));
        }

        private void InitializeAuditLogQueryParams()
        {
            if (_auditLogQueryParams == null)
            {
                _auditLogQueryParams = typeof(AuditLogSearchQuery)
                    .GetMembers()
                    .Where(m => m.MemberType == MemberTypes.Property || m.MemberType == MemberTypes.Field)
                    .Select(p => p.Name)
                    .ToArray();
            }
        }
        private void InitializeChangeLogQueryParams()
        {
            if (_changeLogQueryParams == null)
            {
                _changeLogQueryParams = typeof(ChangeLogSearchQuery)
                    .GetMembers()
                    .Where(m => m.MemberType == MemberTypes.Property || m.MemberType == MemberTypes.Field)
                    .Select(p => p.Name)
                    .ToArray();
            }
        }
    }
}
