﻿using System.Collections.Generic;
using Amazon.SQS;
using Microsoft.AspNetCore.Builder;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.DependencyInjection;
using Microsoft.Extensions.Logging;
using Microsoft.OpenApi.Models;
using Newtonsoft.Json.Serialization;
using Swashbuckle.AspNetCore.Swagger;
using Twitch.AuditLogClient.Extensions;
using Twitch.AuditLogService.ChangeLog;
using Twitch.ElasticSearch.Extensions;
using Twitch.Shared.Web.Extensions;
using Twitch.Shared.Extensions;
using Twitch.Shared.Model;
using Twitch.Shared.Monitoring.Extensions;
using Twitch.Shared.Sqs.Extensions;
using Twitch.Shared.Sqs.MessageProcessors;
using IHostingEnvironment = Microsoft.AspNetCore.Hosting.IHostingEnvironment;

namespace Twitch.AuditLogService.Web
{
    public class Startup
    {
        private readonly IConfiguration _configuration;
        private readonly IHostingEnvironment _hostingEnvironment;
        public Startup(IConfiguration configuration, IHostingEnvironment hostingEnvironment)
        {
            _configuration = configuration;
            _hostingEnvironment = hostingEnvironment;
        }

        // This method gets called by the runtime. Use this method to add services to the container.
        public void ConfigureServices(IServiceCollection services)
        {
            services.AddTwitchDefaults();

            // Register Options
            services.AddDefaultAWSOptions(_configuration.GetAWSOptions());

            services.AddControllers()
                .AddNewtonsoftJson(options =>
                {
                    options.SerializerSettings.ContractResolver = new CamelCasePropertyNamesContractResolver()
                    {
                        NamingStrategy = new CamelCaseNamingStrategy
                        {
                            ProcessDictionaryKeys = false,
                            OverrideSpecifiedNames = true
                        }
                    };
                });

            // Use Light-Weight MvcCore
            ConfigureMvc(services.AddMvcCore());
            services.ConfigureJwtAuth(_configuration);
            services.ConfigureCors(_configuration);

            // Register Service-Specific Dependencies
            services.AddAWSService<IAmazonSQS>();
            services.AddSingleton<IAuditLogSearchManager, AuditLogSearchManager>();
            services.AddSingleton<IChangeLogSearchManager, ChangeLogSearchManager>();

            // Register Tools
            if (!ServiceInfo.IsProduction)
            {
                services.AddSwaggerGen(c => { 
                    c.SwaggerDoc("v1", new OpenApiInfo { Title = ServiceInfo.ServiceName, Version = "v1" }); 

                    c.AddSecurityDefinition("Bearer", //Name the security scheme
                        new OpenApiSecurityScheme
                        {
                            Description = "JWT Authorization header using the Bearer scheme.",
                            Type = SecuritySchemeType.Http, //We set the scheme type to http since we're using bearer authentication
                            Scheme = "bearer" //The name of the HTTP Authorization scheme to be used in the Authorization header. In this case "bearer".
                        });

                    c.AddSecurityRequirement(new OpenApiSecurityRequirement{
                        {
                            new OpenApiSecurityScheme{
                                Reference = new OpenApiReference{
                                    Id = "Bearer", //The name of the previously defined security scheme.
                                    Type = ReferenceType.SecurityScheme
                                }
                            },new List<string>()
                        }
                    });
                });
            }

            services.UseTwitchElastic(_configuration, ServiceInfo.ServiceName);

            services.AddTwitchSqsConsumerAndWorkers(_configuration);
            services.AddTransient<IMessageProcessor, AuditLogMessageProcessor>();
            services.AddTransient<IMessageProcessor, ChangeLogMessageProcessor>();

            services.AddAuditLog(_configuration);
            services.AddHttpClient();

            services.AddTwitchMonitoring(_configuration);
        }

        // This method gets called by the runtime. Use this method to configure the HTTP request pipeline.
        public void Configure(IApplicationBuilder app, IHostingEnvironment env, ILogger<Startup> logger)
        {
            app.UseAuthentication();

            if (!ServiceInfo.IsProduction)
            {
                app.UseSwagger();
                app.UseSwaggerUI(c => c.SwaggerEndpoint("/swagger/v1/swagger.json", $"{ServiceInfo.ServiceName} API V1"));
            }

            app.AddCors();
            app.UseRouting();
            app.UseAuthorization();
            app.UseEndpoints(endpoints => {
                endpoints.MapControllers();
            });

            logger.Information($"{ServiceInfo.ServiceName} ({ServiceInfo.ServiceId}) starting.", ServiceInfo.GetSummaryForLogging());
        }

        private void ConfigureMvc(IMvcCoreBuilder builder)
        {
            // Enable features as needed.
            if (!ServiceInfo.IsProduction)
            {
                // Needed for Swagger
                builder.AddApiExplorer();
            }

            builder.AddAuthorization();
            //builder.AddFormatterMappings();
            //builder.AddViews();
            //builder.AddRazorViewEngine();
            //builder.AddRazorPages();
            //builder.AddCacheTagHelper();
            //builder.AddDataAnnotations();
            builder.AddCors();
        }
    }
}
