﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Microsoft.Extensions.Logging;
using Newtonsoft.Json;
using Twitch.AuditLogService.Models;
using Twitch.Shared.Extensions;
using Twitch.Shared.Sqs.MessageProcessors;
using Twitch.Shared.Sqs.Models;
using Twitch.Shared.Util;

namespace Twitch.AuditLogService
{
    public class AuditLogMessageProcessor : IBatchMessageProcessor
    {
        private readonly IAuditLogSearchManager _auditLogSearchManager;
        private readonly ILogger<AuditLogMessageProcessor> _logger;

        public AuditLogMessageProcessor(ILogger<AuditLogMessageProcessor> logger, IAuditLogSearchManager auditLogSearchManager)
        {
            Ensure.NotNull(nameof(logger), logger);
            _logger = logger;

            Ensure.NotNull(nameof(auditLogSearchManager), auditLogSearchManager);
            _auditLogSearchManager = auditLogSearchManager;
        }

        public async Task<List<WrappedSqsMessage>> TryProcessMessagesAsync(IEnumerable<WrappedSqsMessage> messages, CancellationToken cancellationToken = default(CancellationToken))
        {
            Ensure.NotNull(nameof(messages), messages);
            var processedMessages = new List<WrappedSqsMessage>();
            foreach (var message in messages)
            {
                if (!(message.ParsedMessage is SnsMessage snsMessage))
                {
                    _logger.Warning("Unexpected non-SNS message received", 300_000, message.Message.Body);
                    continue;
                }

                if (snsMessage.SnsMessageType == SnsMessageType.SubscriptionConfirmation)
                {
                    continue;
                }

                try
                {
                    var entry = JsonConvert.DeserializeObject<AuditLogElasticEntry>(snsMessage.Content);
                    
                    if (entry.Created <= 0) // Not a valid audit log entry - possibly Change Log Entry
                    {
                        continue;
                    }

                    entry.TrackedDataChanges = entry.DataChanges?.Select(dc => { dc.Value.Key = dc.Key; return dc.Value; }).ToList();
                    var success = await _auditLogSearchManager.IndexAsync(entry);
                    // Delete
                    if (success)
                    {
                        processedMessages.Add(message);
                    }
                }
                catch (Exception ex)
                {
                    _logger.Error(ex, "Error processing message", new { message });
                }
            }

            return processedMessages;
        }

        public string Name => nameof(AuditLogMessageProcessor);
    }
}
