﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using Microsoft.Extensions.Caching.Memory;
using Microsoft.Extensions.Logging;
using Microsoft.Extensions.Options;
using Nest;
using Twitch.AuditLogService.Models;
using Twitch.ElasticSearch;
using Twitch.ElasticSearch.Extensions;
using Twitch.Shared.Extensions;
using Twitch.Shared.Util;

namespace Twitch.AuditLogService
{
    public sealed class AuditLogSearchManager : CloudSearchManager<AuditLogElasticEntry>, IAuditLogSearchManager
    {
        public AuditLogSearchManager(IElasticClient client,
            IOptions<SearchConfiguration> searchConfigurationAccessor,
            ILogger<AuditLogSearchManager> logger,
            IMemoryCache memoryCache) : base(client, searchConfigurationAccessor, logger, memoryCache)
        {
            EnsureTemplateCreation();
        }

        private const int MaxPageSize = 10_000;
        public const string EmptyFilterString = "none";

        public async Task<bool> IndexAsync(AuditLogElasticEntry item)
        {
            Ensure.NotNull(nameof(item), item);

            var entryDate = DateTimeOffset.FromUnixTimeMilliseconds(item.Created).DateTime;
            var indexName = TimeSeriesIndexing.GetIndexName(_indexTypeName, _indexTimeFrame, entryDate);
            var sendRequest = new IndexRequest<AuditLogElasticEntry>(item, indexName);
            var response = await _client.IndexAsync(sendRequest);
            if (!response.IsValid)
            {
                if (response.OriginalException != null)
                {
                    _logger.Warning(response.OriginalException, "Failed to index document!", response.DebugInformation);
                }
                else
                {
                    _logger.Warning("Failed to index document!", response.DebugInformation);
                }
            }

            return response.IsValid;
        }

        public async Task<IReadOnlyCollection<AuditLogElasticEntry>> SearchAsync(AuditLogSearch searchCriteria, int pageSize)
        {
            // Max page size is 10_000 records, else elasticsearch will error.
            pageSize = Math.Min(pageSize, MaxPageSize);

            var descriptor = (await PrepareSearchDescriptorAsync(searchCriteria))
                .Size(pageSize)
                .Source(s => s.IncludeAll());

            var searchResult = await _client.SearchAsync<AuditLogElasticEntry>(descriptor);

            if (searchResult.IsValid)
            {
                return searchResult.Documents;
            }

            if (searchResult.OriginalException != null)
            {
                _logger.Warning(searchResult.OriginalException, "Failed to search documents", 30_000, searchResult.DebugInformation);
            }
            else
            {
                _logger.Warning("Failed to search documents", 30_000, searchResult.DebugInformation);
            }

            return Array.Empty<AuditLogElasticEntry>();
        }

        public async Task<Dictionary<string, Dictionary<string, long>>> GetTermFiltersAsync(AuditLogSearch searchCriteria)
        {
            var descriptor = (await PrepareSearchDescriptorAsync(searchCriteria))
                .Size(0)
                .Aggregations(a => a
                    .Terms(nameof(AuditLogSearchFilter.Name), t => t
                        .Field(x => x.Name)
                        .Size(5000).Missing(EmptyFilterString))
                    .Terms(nameof(AuditLogSearchFilter.Email), t => t
                        .Field(x => x.Email)
                        .Size(5000).Missing(EmptyFilterString))
                    .Terms(nameof(AuditLogSearchFilter.AmazonUID), t => t
                        .Field(x => x.AmazonUID)
                        .Size(5000).Missing(EmptyFilterString))
                    .Terms(nameof(AuditLogSearchFilter.Service), t => t
                        .Field(x => x.Service)
                        .Size(5000).Missing(EmptyFilterString))
                    .Terms(nameof(AuditLogSearchFilter.Handler), t => t
                        .Field(x => x.Handler.Suffix("keyword"))
                        .Size(5000).Missing(EmptyFilterString))
                    .Terms(nameof(AuditLogSearchFilter.Uri), t => t
                        .Field(x => x.Uri.Suffix("keyword"))
                        .Size(5000).Missing(EmptyFilterString))
                    .Terms(nameof(AuditLogSearchFilter.Method), t => t
                        .Field(x => x.Method)
                        .Size(20).Missing(EmptyFilterString))
                    .Terms(nameof(AuditLogSearchFilter.ResponseCode), t => t
                        .Field(x => x.ResponseCode)
                        .Size(100))
                    .Terms(nameof(AuditLogSearchFilter.Canceled), t => t
                        .Field(x => x.Canceled)
                        .Size(3))
                    .Terms(nameof(AuditLogSearchFilter.Exception), t => t
                        .Field(x => x.Exception)
                        .Size(3))
                    .Terms(nameof(AuditLogSearchFilter.ActionType), t => t
                        .Field(x => x.ActionType)
                        .Size(100).Missing(EmptyFilterString))
                    .Terms(nameof(AuditLogSearchFilter.OperationName), t => t
                        .Field(x => x.OperationName)
                        .Size(100).Missing(EmptyFilterString)));
                      
            var res = await _client.SearchAsync<AuditLogElasticEntry>(descriptor);
            var result = new Dictionary<string, Dictionary<string, long>>();
            if (!res.IsValid)
            {
                if (res.OriginalException != null)
                {
                    _logger.Warning(res.OriginalException, "Failed to get term filters", 30_000, res.DebugInformation);
                }
                else
                {
                    _logger.Warning("Failed to get term filters", 30_000, res.DebugInformation);
                }

                return result;
            }

            foreach (var aggregation in res.Aggregations.OrderBy(a => a.Key))
            {
                var innerResult = new Dictionary<string, long>();
                foreach (var val in res.Aggs.Terms(aggregation.Key).Buckets)
                {
                    var key = val.KeyAsString ?? val.Key;
                    if (string.IsNullOrEmpty(key))
                    {
                        key = EmptyFilterString;
                    }

                    var count = val.DocCount.GetValueOrDefault(0);
                    if (innerResult.ContainsKey(key))
                    {
                        count += innerResult[key];
                    }
                    innerResult[key] = count;
                }

                result[aggregation.Key] = innerResult;
            }

            return result;
        }

        private async Task<SearchDescriptor<AuditLogElasticEntry>> PrepareSearchDescriptorAsync(AuditLogSearch searchCriteria)
        {
            List<string> indexNames;
            if (searchCriteria == null)
            {
                indexNames = await GetIndexNamesAsync(DateTime.UtcNow.AddDays(-1));
                return new SearchDescriptor<AuditLogElasticEntry>()
                    .Index(Indices.Index(indexNames));
            }

            DateTime startDate;
            DateTime endDate;
            if (searchCriteria.SearchFilter?.Timestamp != null)
            {
                // We could be searching in either direction, so we add 1 day to both direction as the range
                var parsedDate = DateTimeOffset.FromUnixTimeMilliseconds(searchCriteria.SearchFilter.Timestamp).UtcDateTime;
                startDate = parsedDate.AddDays(-1);
                endDate = parsedDate.AddDays(1);
            }
            else
            {
                endDate = DateTime.UtcNow;
                startDate = endDate.AddDays(-1);
            }

            indexNames = await GetIndexNamesAsync(startDate, endDate);

            return new SearchDescriptor<AuditLogElasticEntry>()
                .ApplyScrollingSearchFilter(searchCriteria.SearchFilter, x => x.Created, x => x.Id)
                .Index(Indices.Index(indexNames))
                .Query(searchCriteria.GetQuery);
        }
    }
}