﻿using System;
using System.Collections.Generic;
using System.Threading;
using System.Threading.Tasks;
using Microsoft.Extensions.Logging;
using Newtonsoft.Json;
using Twitch.AuditLogService.ChangeLog.Models;
using Twitch.Shared.Extensions;
using Twitch.Shared.Sqs.MessageProcessors;
using Twitch.Shared.Sqs.Models;
using Twitch.Shared.Util;

namespace Twitch.AuditLogService.ChangeLog
{
    public class ChangeLogMessageProcessor : IBatchMessageProcessor
    {
        private const long MaxUnixDateTimeSeconds = 253402300799;
        private readonly IChangeLogSearchManager _changeLogSearchManager;
        private readonly ILogger<ChangeLogMessageProcessor> _logger;

        public ChangeLogMessageProcessor(ILogger<ChangeLogMessageProcessor> logger, IChangeLogSearchManager changeLogSearchManager)
        {
            Ensure.NotNull(nameof(logger), logger);
            _logger = logger;

            Ensure.NotNull(nameof(changeLogSearchManager), changeLogSearchManager);
            _changeLogSearchManager = changeLogSearchManager;
        }

        public async Task<List<WrappedSqsMessage>> TryProcessMessagesAsync(IEnumerable<WrappedSqsMessage> messages, CancellationToken cancellationToken = default(CancellationToken))
        {
            Ensure.NotNull(nameof(messages), messages);
            var processedMessages = new List<WrappedSqsMessage>();
            foreach (var message in messages)
            {
                if (!(message.ParsedMessage is SnsMessage snsMessage))
                {
                    _logger.Warning("Unexpected non-SNS message received", 300_000, message.Message.Body);
                    continue;
                }

                if (snsMessage.SnsMessageType == SnsMessageType.SubscriptionConfirmation)
                {
                    continue;
                }

                try
                {
                    var entry = JsonConvert.DeserializeObject<ChangeLogElasticEntry>(snsMessage.Content);

                    if (entry.Timestamp <= 0) // Not a valid change log entry - possibly Audit Log Entry
                    {
                        continue;
                    }

                    if (entry.Timestamp <= MaxUnixDateTimeSeconds)
                    {
                        // Update Entry to use millisecond precision instead of seconds
                        entry.Timestamp = entry.Timestamp * 1000;
                    }

                    var success = await _changeLogSearchManager.IndexAsync(entry);
                    // Delete
                    if (success)
                    {
                        processedMessages.Add(message);
                    }
                }
                catch (Exception ex)
                {
                    _logger.Error(ex, "Error processing message", new { message });
                }
            }

            return processedMessages;
        }

        public string Name => nameof(ChangeLogMessageProcessor);
    }
}
