﻿using System.Collections.Generic;
using Nest;
using Twitch.ElasticSearch;

namespace Twitch.AuditLogService.Models
{
    [CloudSearchModel(IndexTypeName = "audit-log", UseDefaultIndex = false, AutoCreateIndex = false, QueryCacheEnabled = true, ShardCount = 2, TemplateVersion = 2)]
    [ElasticsearchType(IdProperty = "Id", Name = "auditLog")]
    public class AuditLogElasticEntry
    {
        /// <summary>
        /// Unique Id generated for this entry.
        /// </summary>
        [Keyword]
        public string Id { get; set; }

        /// <summary>
        /// Miliseconds since unix epoch.
        /// </summary>
        [Date]
        public long Created { get; set; }

        // Who
        /// <summary>
        /// JTI unique identifier of the Token.
        /// </summary>
        [Keyword]
        public string TokenId { get; set; }

        /// <summary>
        /// Name of the user.
        /// </summary>
        /// <remarks>"cn" from LDAP.</remarks>
        [Keyword]
        public string Name { get; set; }

        /// <summary>
        /// Email address of the user.
        /// </summary>
        /// <remarks>"mail" from LDAP.</remarks>
        [Keyword]
        public string Email { get; set; }

        /// <summary>
        /// User Id.
        /// </summary>
        /// <remarks>"twAmazonUID" from LDAP.</remarks>
        [Keyword]
        public string AmazonUID { get; set; }

        /// <summary>
        /// Manager the user reports to, if available.
        /// </summary>
        /// <remarks>"manager" from LDAP.</remarks>
        [Text]
        public string Manager { get; set; }

        /// <summary>
        /// Title of the user, if available.
        /// </summary>
        /// <remarks>"title" from LDAP.</remarks>
        [Text]
        public string Title { get; set; }

        /// <summary>
        /// Department the user belongs to, if available.
        /// </summary>
        /// <remarks>"department" from LDAP.</remarks>
        [Text]
        public string Department { get; set; }

        /// <summary>
        /// IP Address of the user.
        /// </summary>
        [Ip]
        public string ClientIP { get; set; }

        /// <summary>
        /// User-Agent of the user.
        /// </summary>
        [Text]
        public string UserAgent { get; set; }

        // What
        /// <summary>
        /// The name of the service.
        /// </summary>
        [Keyword]
        public string Service { get; set; }

        /// <summary>
        /// The controller handler name. ( Controller + method )
        /// </summary>
        public string Handler { get; set; }

        /// <summary>
        /// The request Uri. 
        /// </summary>
        public string Uri { get; set; }

        /// <summary>
        /// The request Method.
        /// </summary>
        [Keyword]
        public string Method { get; set; }

        /// <summary>
        /// Data passed to the controller.
        /// </summary>
        [Text]
        public string Data { get; set; }

        /// <summary>
        /// The roles permitted to execute this request.
        /// </summary>
        [Text]
        public string RolesAllowed { get; set; }

        // Response
        /// <summary>
        /// The Http Status code response.
        /// </summary>
        [Keyword]
        public int ResponseCode { get; set; }

        /// <summary>
        /// Flag indicating whether the request was cancelled mid-flight.
        /// </summary>
        [Boolean]
        public bool Canceled { get; set; }

        /// <summary>
        /// Flag indicating whether the request was aborted mid-way due to a Exception.
        /// </summary>
        [Boolean]
        public bool Exception { get; set; }

        /// <summary>
        /// Type of the request. e.g. read or write.
        /// </summary>
        [Keyword]
        public string ActionType { get; set; }

        /// <summary>
        /// Friendly name of the operation being performed.
        /// </summary>
        [Keyword]
        public string OperationName { get; set; }

        /// <summary>
        /// Dictionary of data changes reported during this request.
        /// </summary>
        [Nested(Ignore = true)]
        public Dictionary<string, AuditLogElasticDataChange> DataChanges { get; set; }

        [Nested]
        public List<AuditLogElasticDataChange> TrackedDataChanges { get; set; }
    }
}