package audit

import (
	"context"
	"encoding/json"
	"fmt"
	"sync"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
	"github.com/sirupsen/logrus"
)

// Client is a client to interact directly with the dynamo tables.
type Client struct {
	Environment   string
	DeleteThreads int
	Logger        logrus.FieldLogger
	ScanBatchSize int64

	initOnce  sync.Once
	db        dynamodbiface.DynamoDBAPI
	tableName string
}

// initialize the client. all public methods should call this to make sure
// clients are initialized.
func (c *Client) init() (err error) {
	c.initOnce.Do(func() {
		var roleARN string
		switch c.Environment {
		case "":
			fallthrough
		case "prod":
			c.tableName = "history-prod"
			roleARN = "arn:aws:iam::958416494912:role/history-admin-prod"
		case "staging":
			c.tableName = "history-staging"
			roleARN = "arn:aws:iam::005087123760:role/history-admin-staging"
		default:
			err = fmt.Errorf("invalid audit environment: %s", c.Environment)
			return
		}

		if c.ScanBatchSize == 0 {
			c.ScanBatchSize = 10000
		}

		c.db = dynamodb.New(session.Must(session.NewSession(&aws.Config{
			Region: aws.String("us-west-2"),
			Credentials: stscreds.NewCredentials(session.Must(session.NewSession(&aws.Config{
				Region: aws.String("us-west-2"),
			})), roleARN),
		})))
	})
	return
}

// ScanAuditsHandler is the item callback for ScanAudits
type ScanAuditsHandler func(*Audit) (cont bool)

// ScanAudits runs a handler on all entries of the audits dynamo table
func (c *Client) ScanAudits(
	ctx context.Context,
	h ScanAuditsHandler,
) error {
	if err := c.init(); err != nil {
		return err
	}

	var pageError error
	if err := c.db.ScanPagesWithContext(ctx, &dynamodb.ScanInput{
		Limit:     aws.Int64(c.ScanBatchSize),
		TableName: aws.String(c.tableName),
	}, func(output *dynamodb.ScanOutput, lastPage bool) bool {
		for _, item := range output.Items {
			var audit *Audit
			if audit, pageError = auditFromRow(item); pageError != nil {
				return false
			}

			var cont bool
			if cont = h(audit); !cont {
				return false
			}
		}
		return !lastPage
	}); err != nil {
		return err
	}
	return pageError
}

// ScanAuditsPooled runs a handler on all entries of the audits dynamo table
func (c *Client) ScanAuditsPooled(
	ctx context.Context,
	poolSize int,
	h ScanAuditsHandler,
) error {
	if err := c.init(); err != nil {
		return err
	}

	stopChan := make(chan interface{})
	poolChan := make(chan interface{}, poolSize)

	return c.db.ScanPagesWithContext(ctx, &dynamodb.ScanInput{
		Limit:     aws.Int64(c.ScanBatchSize),
		TableName: aws.String(c.tableName),
	}, func(output *dynamodb.ScanOutput, lastPage bool) bool {
		for _, item := range output.Items {
			item := item

			select {
			case <-stopChan:
				return false
			case poolChan <- nil:
			}

			go func() {
				audit, err := auditFromRow(item)
				if err != nil {
					c.Logger.Error(err)
					stopChan <- nil
					return
				}

				if cont := h(audit); !cont {
					stopChan <- nil
					return
				}

				<-poolChan
			}()
		}
		return !lastPage
	})
}

func auditFromRow(item map[string]*dynamodb.AttributeValue) (*Audit, error) {
	dynamoAudit := struct {
		Body string `dynamodbav:"body"`
	}{}

	if err := dynamodbattribute.UnmarshalMap(item, &dynamoAudit); err != nil {
		return nil, err
	}

	audit := new(Audit)
	if err := json.Unmarshal([]byte(dynamoAudit.Body), audit); err != nil {
		return nil, err
	}

	return audit, nil
}

// IDStream is a stream of audit ids
type IDStream interface {
	Next(context.Context) (id string, last bool)
}

// DeleteAudits ...
func (c *Client) DeleteAudits(ctx context.Context, ids IDStream) error {
	if err := c.init(); err != nil {
		return err
	}

	batches := &deleteBatchStream{
		IDStream: ids,
		Size:     25,
	}

	deletePool := &pool{Size: c.DeleteThreads}
	for {
		batch, last := batches.Next(ctx)
		if len(batch) > 0 {
			deletePool.Do(func() {
				c.deleteAudits(ctx, batch)
			})
		}

		if last {
			break
		}
	}

	return nil
}

func (c *Client) deleteAudits(ctx context.Context, batch []*dynamodb.WriteRequest) {
	nBackoff := 0
	for len(batch) > 0 {
		time.Sleep(time.Duration(2*nBackoff) * 500 * time.Millisecond)

		res, err := c.db.BatchWriteItemWithContext(ctx, &dynamodb.BatchWriteItemInput{
			RequestItems: map[string][]*dynamodb.WriteRequest{c.tableName: batch},
		})

		if err != nil {
			c.Logger.Error(err)
		} else {
			batch = res.UnprocessedItems[c.tableName]
		}

		if len(batch) > 0 {
			c.Logger.Infof("retrying %d items", len(batch))
		}

		nBackoff++
	}
}
