package historycmd

import (
	"context"
	"encoding/csv"
	"encoding/json"
	"errors"
	"flag"
	"io"
	"os"

	"github.com/google/subcommands"
	elastic "gopkg.in/olivere/elastic.v5"
)

// DumpActionCmd ...
type DumpActionCmd struct {
	*BaseCmd

	elastic *elastic.Client
	action  string
}

// Name implements subcommands
func (cmd DumpActionCmd) Name() string {
	return "dump-action"
}

// Synopsis implements subcommands
func (cmd DumpActionCmd) Synopsis() string {
	return "dump action history records as csv"
}

// Usage implements subcommands
func (cmd DumpActionCmd) Usage() string {
	return `dump-action -action [action] > out.csv
`
}

// SetFlags implements subcommands
func (cmd *DumpActionCmd) SetFlags(f *flag.FlagSet) {
	f.StringVar(&cmd.action, "action", "", "action to dump.")
}

// Execute implements subcommands
func (cmd DumpActionCmd) Execute(ctx context.Context, f *flag.FlagSet, args ...interface{}) subcommands.ExitStatus {
	if err := cmd.init(ctx); err != nil {
		cmd.Logger.Error(err)
		return subcommands.ExitFailure
	}

	if err := cmd.execute(ctx); err != nil {
		cmd.Logger.Error(err)
		return subcommands.ExitFailure
	}

	return subcommands.ExitSuccess
}

func (cmd *DumpActionCmd) init(ctx context.Context) error {
	if cmd.action == "" {
		return errors.New("-action is required")
	}

	elasticSearchClient, err := cmd.elasticSearchClient()
	if err != nil {
		return err
	}
	cmd.elastic = elasticSearchClient
	return nil
}

func (cmd DumpActionCmd) execute(ctx context.Context) error {
	scroll := cmd.elastic.Scroll().
		Index("history-*").
		Type("audits").
		Query(elastic.NewBoolQuery().
			Filter(elastic.NewTermsQuery("action", cmd.action))).
		Size(100)

	w := csv.NewWriter(os.Stdout)
	defer w.Flush()

	if err := w.Write([]string{
		"created_at",
		"action",
		"resource_type",
		"resource_id",
	}); err != nil {
		return err
	}

	wroteItem := cmd.logExecutionCount("wrote-item", 1000)
	for {
		page, err := scroll.Do(ctx)
		if err == io.EOF {
			return nil
		} else if err != nil {
			return err
		}

		for _, hit := range page.Hits.Hits {
			wroteItem()

			audit := struct {
				Action       string `json:"action"`
				CreatedAt    string `json:"created_at"`
				ResourceType string `json:"resource_type"`
				ResourceID   string `json:"resource_id"`
			}{}

			if err := json.Unmarshal(*hit.Source, &audit); err != nil {
				return err
			}

			if err := w.Write([]string{
				audit.CreatedAt,
				audit.Action,
				audit.ResourceType,
				audit.ResourceID,
			}); err != nil {
				return err
			}
		}
	}
}
