resource "aws_ecs_cluster" "edge" {
  name = "${local.resource_prefix}"
}

resource "aws_ecs_service" "edge" {
  name            = "${local.resource_prefix}"
  cluster         = "${aws_ecs_cluster.edge.arn}"
  task_definition = "${aws_ecs_task_definition.edge.arn}"
  desired_count   = "${var.desired_count}"
  launch_type     = "FARGATE"

  load_balancer {
    target_group_arn = "${aws_lb_target_group.edge.arn}"
    container_name   = "${local.resource_prefix}"
    container_port   = "8000"
  }

  network_configuration {
    security_groups = [
      "${var.twitch_security_groups}",
    ]

    subnets = [
      "${var.twitch_subnets}",
    ]
  }

  lifecycle {
    ignore_changes = [
      "task_definition",
      "desired_count",
    ]
  }

  depends_on = [
    "aws_lb_listener.edge",
  ]
}

resource "aws_ecs_task_definition" "edge" {
  family                   = "${local.resource_prefix}"
  network_mode             = "awsvpc"
  requires_compatibilities = ["FARGATE"]
  cpu                      = "1024"
  memory                   = "2048"

  execution_role_arn = "${aws_iam_role.execution.arn}"
  task_role_arn      = "${aws_iam_role.task.arn}"

  lifecycle {
    ignore_changes = [
      "container_definition",
    ]
  }

  container_definitions = <<DEFINITION
[
  {
    "cpu": 1024,
    "memory": 2048,
    "essential": true,
    "image": "${aws_ecr_repository.edge.repository_url}:latest",
    "name": "${local.resource_prefix}",
    "networkMode": "awsvpc",
    "logConfiguration": {
      "logDriver": "awslogs",
      "options": {
        "awslogs-group": "${aws_cloudwatch_log_group.edge.name}",
        "awslogs-region": "${var.region}",
        "awslogs-stream-prefix": "${local.resource_prefix}"
      }
    },

    "environment": [
        {
            "name": "ENVIRONMENT",
            "value": "${var.environment}"
        },
        {
            "name": "ServiceName",
            "value": "history-edge"
        }
    ],
    "portMappings": [
      {
        "containerPort": 8000
      }
    ]
  }
]
  DEFINITION
}

resource "aws_cloudwatch_log_group" "edge" {
  name = "history-edge/${var.environment}"

  tags {
    Service     = "history-edge"
    Environment = "${var.environment}"
  }
}
