package main

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"log"
	"os"

	"code.justin.tv/foundation/history-service/internal/gdpr"
	"code.justin.tv/foundation/history-service/internal/queue"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/aws/aws-sdk-go/service/sqs/sqsiface"
)

func main() {
	region, ok := os.LookupEnv("REGION")
	if !ok {
		log.Fatal("environment variable 'REGION' is not set")
	}

	environment, ok := os.LookupEnv("ENVIRONMENT")
	if !ok {
		log.Fatal("environment variable 'ENVIRONMENT' is not set")
	}

	queueURL, ok := os.LookupEnv("QUEUE_URL")
	if !ok {
		log.Fatal("environment variable 'QUEUE_URL' is not set")
	}

	da := &deleteAudits{
		queue: sqs.New(session.Must(session.NewSession(&aws.Config{
			Region: aws.String(region),
		}))),
		queueURL: queueURL,
		gdprClient: &gdpr.Client{
			Environment: environment,
		},
	}

	err := da.DeleteWithContext(context.Background())
	if err != nil {
		log.Fatal(err)
	}
	log.Println("finished processsing delete user audits requests")
}

// deleteAudits processes the queue and provides methods to delete user audits
// from dynamodb and elastic search
type deleteAudits struct {
	queue      sqsiface.SQSAPI
	gdprClient gdpr.ClientAPI
	queueURL   string
}

// delete user audits from dynamodb and elastic search service
func (d *deleteAudits) DeleteWithContext(ctx context.Context) error {

	for {
		log.Printf("retrieving delete user audit request from queue: %s", d.queueURL)
		resp, err := d.queue.ReceiveMessageWithContext(
			ctx,
			&sqs.ReceiveMessageInput{
				QueueUrl:            aws.String(d.queueURL),
				MaxNumberOfMessages: aws.Int64(1),
				WaitTimeSeconds:     aws.Int64(20),
			},
		)
		if err != nil {
			return err
		}

		if len(resp.Messages) == 0 {
			log.Printf("no message found in sqs: %s, exiting", d.queueURL)
			return nil
		}

		sqsMessage := resp.Messages[0]
		msg := &queue.DeleteUserMessage{}
		err = json.Unmarshal([]byte(*sqsMessage.Body), msg)
		if err != nil {
			return errors.New("error parsing SQS message: " + err.Error())
		}

		// Process message
		log.Printf("deleting all user audit events for user: %s from dynamodb and ES. this may take a while", msg.UserID)

		err = d.gdprClient.DeleteAudits(ctx, msg.UserID)
		if err != nil {
			return err
		}
		log.Printf("finished deleting user data, deleting processed request from request queue")
		//delete the message when done.
		_, err = d.queue.DeleteMessageWithContext(ctx, &sqs.DeleteMessageInput{
			ReceiptHandle: sqsMessage.ReceiptHandle,
			QueueUrl:      aws.String(d.queueURL),
		})
		if err != nil {
			return fmt.Errorf("error deleting SQS Message(%s): %s", sqsMessage, err.Error())
		}
	}
}
