package ecs

import (
	"context"
	"errors"
	"flag"
	"fmt"

	"code.justin.tv/foundation/history-service/internal/ecstasks/definition"
	"github.com/google/subcommands"
	"github.com/sirupsen/logrus"
)

// UpdateTaskDef ....
type UpdateTaskDef struct {
	gitCommit     string
	environment   string
	stageMetadata *stageMetadata
	Logger        logrus.FieldLogger
}

// Name implement subcommands
func (svc *UpdateTaskDef) Name() string {
	return "ecs"
}

// Synopsis implement subcommands
func (svc *UpdateTaskDef) Synopsis() string {
	return "update all ecs task"
}

// Usage implement subcommands
func (svc *UpdateTaskDef) Usage() string {
	return `ecs -env <'prod', or 'staging'> -git-commit <commit hash to be deployed on ecs>
	`
}

// SetFlags implement subcommands
func (svc *UpdateTaskDef) SetFlags(f *flag.FlagSet) {
	f.StringVar(&svc.environment, "env", "", "'staging'or 'prod'")
	f.StringVar(&svc.gitCommit, "git-commit", "", "git commit hash which need to be deployed to ecs")
}

//Execute implement subcommands
func (svc *UpdateTaskDef) Execute(_ context.Context, f *flag.FlagSet, _ ...interface{}) subcommands.ExitStatus {

	if err := svc.init(); err != nil {
		svc.Logger.Error(err.Error())
		return subcommands.ExitFailure
	}

	err := svc.execute()
	if err != nil {
		svc.Logger.Error(err.Error())
		return subcommands.ExitFailure
	}
	return subcommands.ExitSuccess
}

type stageMetadata struct {
	serviceType   string
	servicePrefix string
}

func (svc *UpdateTaskDef) getStageMetadata() (*stageMetadata, error) {

	if svc.environment == "" {
		return nil, errors.New("environment required")
	}

	stagingMetadata := &stageMetadata{
		serviceType: "staging",
	}

	prodMetadata := &stageMetadata{
		serviceType: "prod",
	}

	switch svc.environment {
	case "staging":
		stagingMetadata.servicePrefix = "history-service-v3"
		return stagingMetadata, nil
	case "prod":
		prodMetadata.servicePrefix = "history-service-v3"
		return prodMetadata, nil
	default:
		return nil, fmt.Errorf("invalid stage '%s' provided", svc.environment)
	}
}

func (svc *UpdateTaskDef) init() error {

	if svc.environment == "" {
		return errors.New("ENVIRONMENT must be set")
	}

	if svc.gitCommit == "" {
		return errors.New("GIT_COMMIT must be set")
	}

	var err error
	svc.stageMetadata, err = svc.getStageMetadata()
	if err != nil {
		return err
	}

	return nil
}

func (svc *UpdateTaskDef) execute() error {
	svc.Logger.Info("updating gpdr tasks")
	baseTaskDef := &definition.BaseUpdateTaskDef{
		Environment:    svc.environment,
		GitCommit:      svc.gitCommit,
		ResourcePrefix: svc.stageMetadata.servicePrefix,
	}
	taskDeleteUser := &definition.DeleteUserTaskDef{
		BaseUpdateTaskDef: baseTaskDef,
	}

	registeredTask, err := taskDeleteUser.Register()
	if err != nil {
		return err
	}

	svc.Logger.Infof("updated task %s", *registeredTask.TaskDefinition.Family)

	taskUserReport := &definition.GenerateUserReportTaskDef{
		BaseUpdateTaskDef: baseTaskDef,
	}

	registeredTask, err = taskUserReport.Register()
	if err != nil {
		return err
	}
	svc.Logger.Infof("updated task %s", *registeredTask.TaskDefinition.Family)

	return nil
}
