package lambda

import (
	"context"
	"flag"
	"fmt"
	"os"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3/s3manager"
	"github.com/aws/aws-sdk-go/service/s3/s3manager/s3manageriface"
	"github.com/google/subcommands"
	"github.com/sirupsen/logrus"
)

// S3UploadLambdaFuncCmd upload lambda source to s3
type S3UploadLambdaFuncCmd struct {
	environment string
	sourceDir   string
	gitCommit   string
	uploader    s3manageriface.UploaderAPI
	Logger      logrus.FieldLogger
}

// Name implement subcommands
func (c *S3UploadLambdaFuncCmd) Name() string {
	return "s3-upload-lambda"
}

// Synopsis implement subcommands
func (c *S3UploadLambdaFuncCmd) Synopsis() string {
	return "upload lambda source code to s3 for deployment"
}

// Usage implement subcommands
func (c *S3UploadLambdaFuncCmd) Usage() string {
	return `s3-upload-lambda -env <staging', or 'prod'> -dir <source dir containing all zip files with lambda handler>
	`
}

// SetFlags implement subcommands
func (c *S3UploadLambdaFuncCmd) SetFlags(f *flag.FlagSet) {
	f.StringVar(&c.environment, "env", "", "'staging', or 'prod'")
	f.StringVar(&c.sourceDir, "dir", "", "dir with all lambda handler zip files")
	f.StringVar(&c.gitCommit, "git-commit", "", "git commit hash which need to be deployed to ecs")
}

//Execute implement subcommands
func (c *S3UploadLambdaFuncCmd) Execute(_ context.Context, f *flag.FlagSet, _ ...interface{}) subcommands.ExitStatus {
	err := c.execute()
	if err != nil {
		c.Logger.Error(err.Error())
		return subcommands.ExitFailure
	}
	return subcommands.ExitSuccess
}

func (c *S3UploadLambdaFuncCmd) execute() error {

	c.uploader = s3manager.NewUploader(session.Must(session.NewSession(&aws.Config{
		Region: aws.String("us-west-2"),
	})))

	lambdas, err := getAllLambdaMetadata(c.environment, c.gitCommit)
	if err != nil {
		return err
	}
	c.Logger.Info("Updating lambdas")
	for _, l := range lambdas {
		sourceFile := fmt.Sprintf("%s/%s", c.sourceDir, l.S3Key)
		c.Logger.Infof("uploading source %s to s3 bucket %s/%s", sourceFile, l.S3Bucket, l.S3Key)

		f, err := os.Open(sourceFile)
		if err != nil {
			return fmt.Errorf("failed to open file %q, %v", sourceFile, err)
		}
		// Upload the file to S3.
		_, err = c.uploader.Upload(&s3manager.UploadInput{
			Bucket: aws.String(l.S3Bucket),
			Key:    aws.String(l.S3Key),
			Body:   f,
		})
		if err != nil {
			return fmt.Errorf("failed to upload file, %v", err)
		}
	}
	return nil
}

type functionMetadata struct {
	Name     string
	S3Bucket string
	S3Key    string
}

func getAllLambdaMetadata(environment, gitCommit string) ([]*functionMetadata, error) {
	var meta []*functionMetadata

	var infraPrefix, servicePrefix, serviceType string

	switch environment {
	case "staging":
		infraPrefix = "history-v3"
		servicePrefix = "history-service-v3"
		serviceType = "staging"
	case "prod":
		infraPrefix = "history-v3"
		servicePrefix = "history-service-v3"
		serviceType = "prod"
	default:
		return nil, fmt.Errorf("invalid lambda environment: %s", environment)
	}
	meta = append(meta, &functionMetadata{
		Name:     fmt.Sprintf("%s-%s-stream-ingest", infraPrefix, serviceType),
		S3Bucket: fmt.Sprintf("%s-%s-lambdas", servicePrefix, serviceType),
		S3Key:    fmt.Sprintf("stream-ingest-%s.zip", gitCommit),
	})

	functionNamesSuffix := []string{
		"delete-user",
		"generate-user-report",
		"hard-delete-event-handler",
	}

	for _, fnSuffix := range functionNamesSuffix {
		meta = append(meta, &functionMetadata{
			Name:     fmt.Sprintf("%s-%s-%s", servicePrefix, serviceType, fnSuffix),
			S3Bucket: fmt.Sprintf("%s-%s-lambdas", servicePrefix, serviceType),
			S3Key:    fmt.Sprintf("%s-%s.zip", fnSuffix, gitCommit),
		})
	}
	return meta, nil
}
