package definition

import (
	"errors"
	"fmt"
	"os"
	"sync"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/ecs"
	"github.com/aws/aws-sdk-go/service/ecs/ecsiface"
)

var (
	regionUsWest2    = "us-west-2"
	historyAccountID = map[string]string{
		"staging": "005087123760",
		"prod":    "958416494912",
	}
)

// BaseUpdateTaskDef ....
type BaseUpdateTaskDef struct {
	Environment    string
	ResourcePrefix string
	GitCommit      string

	ecs       ecsiface.ECSAPI
	initOnce  sync.Once
	initError error
}

func (btd *BaseUpdateTaskDef) init() error {
	btd.initOnce.Do(func() {
		if btd.Environment == "" {
			btd.initError = errors.New("ENVIRONMENT must be set")
		}
		if btd.ResourcePrefix == "" {
			btd.initError = errors.New("Resource Prefix must be set")
		}

		btd.GitCommit = os.Getenv("GIT_COMMIT")
		if btd.GitCommit == "" {
			btd.initError = errors.New("GIT_COMMIT must be set")
		}

		btd.ecs = ecs.New(session.Must(session.NewSession(&aws.Config{
			Region: aws.String(regionUsWest2),
		})))
	})

	return btd.initError
}

func (btd *BaseUpdateTaskDef) accountID() string {
	return historyAccountID[btd.getEnvironmentType()]
}

func (btd *BaseUpdateTaskDef) resourcePrefix() string {
	return fmt.Sprintf("%s-%s", btd.ResourcePrefix, btd.getEnvironmentType())
}

func (btd *BaseUpdateTaskDef) getEnvironmentType() string {
	switch btd.Environment {
	case "staging":
		return "staging"
	case "prod":
		return "prod"
	}
	return ""
}
