package definition

import (
	"fmt"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/ecs"
)

// DeleteUserTaskDef ....
type DeleteUserTaskDef struct {
	*BaseUpdateTaskDef
}

func (dutd *DeleteUserTaskDef) getExecuteRoleArn() *string {
	return aws.String(fmt.Sprintf("arn:aws:iam::%s:role/%s-execute-delete-user", dutd.accountID(), dutd.resourcePrefix()))
}

func (dutd *DeleteUserTaskDef) getTaskRoleArn() *string {
	return aws.String(fmt.Sprintf("arn:aws:iam::%s:role/%s-task-delete-user", dutd.accountID(), dutd.resourcePrefix()))
}

func (dutd *DeleteUserTaskDef) image() *string {
	return aws.String(fmt.Sprintf("%s.dkr.ecr.%s.amazonaws.com/%s:delete-user-%s", dutd.accountID(), regionUsWest2, dutd.resourcePrefix(), dutd.GitCommit))
}

func (dutd *DeleteUserTaskDef) queueURL() *string {
	return aws.String(fmt.Sprintf("https://sqs.us-west-2.amazonaws.com/%s/%s-delete-user", dutd.accountID(), dutd.resourcePrefix()))
}

func (dutd *DeleteUserTaskDef) containerDefinition() *ecs.ContainerDefinition {
	return &ecs.ContainerDefinition{
		Image:     dutd.image(),
		Cpu:       aws.Int64(1024),
		Essential: aws.Bool(true),
		Memory:    aws.Int64(2048),
		Name:      aws.String(fmt.Sprintf("%s-delete-user", dutd.resourcePrefix())),
		LogConfiguration: &ecs.LogConfiguration{
			LogDriver: aws.String("awslogs"),
			Options: map[string]*string{
				"awslogs-group":         aws.String(fmt.Sprintf("%s/gdpr", dutd.resourcePrefix())),
				"awslogs-region":        aws.String(regionUsWest2),
				"awslogs-stream-prefix": aws.String("ecs-task-delete-user"),
			},
		},
		Environment: []*ecs.KeyValuePair{
			{
				Name:  aws.String("ENVIRONMENT"),
				Value: aws.String(dutd.Environment),
			},
			{
				Name:  aws.String("REGION"),
				Value: aws.String(regionUsWest2),
			},
			{
				Name:  aws.String("QUEUE_URL"),
				Value: dutd.queueURL(),
			},
		},
	}
}

func (dutd *DeleteUserTaskDef) taskDefinition() *ecs.RegisterTaskDefinitionInput {
	return &ecs.RegisterTaskDefinitionInput{
		ContainerDefinitions: []*ecs.ContainerDefinition{
			dutd.containerDefinition(),
		},
		Cpu:              aws.String("1024"),
		Memory:           aws.String("2048"),
		Family:           aws.String(fmt.Sprintf("%s-delete-user", dutd.resourcePrefix())),
		ExecutionRoleArn: dutd.getExecuteRoleArn(),
		NetworkMode:      aws.String("awsvpc"),
		TaskRoleArn:      dutd.getTaskRoleArn(),
	}
}

// Register the task definition
func (dutd *DeleteUserTaskDef) Register() (*ecs.RegisterTaskDefinitionOutput, error) {
	if err := dutd.init(); err != nil {
		return nil, err
	}
	return dutd.ecs.RegisterTaskDefinition(dutd.taskDefinition())
}
