package definition

import (
	"fmt"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/ecs"
)

// GenerateUserReportTaskDef ....
type GenerateUserReportTaskDef struct {
	*BaseUpdateTaskDef
}

func (gurtd *GenerateUserReportTaskDef) getExecuteRoleArn() *string {
	return aws.String(fmt.Sprintf("arn:aws:iam::%s:role/%s-execute-generate-user-report", gurtd.accountID(), gurtd.resourcePrefix()))
}

func (gurtd *GenerateUserReportTaskDef) getTaskRoleArn() *string {
	return aws.String(fmt.Sprintf("arn:aws:iam::%s:role/%s-task-generate-user-report", gurtd.accountID(), gurtd.resourcePrefix()))
}

func (gurtd *GenerateUserReportTaskDef) image() *string {
	return aws.String(fmt.Sprintf("%s.dkr.ecr.%s.amazonaws.com/%s:generate-user-report-%s", gurtd.accountID(), regionUsWest2, gurtd.resourcePrefix(), gurtd.GitCommit))
}

func (gurtd *GenerateUserReportTaskDef) queueURL() *string {
	return aws.String(fmt.Sprintf("https://sqs.us-west-2.amazonaws.com/%s/%s-generate-user-report", gurtd.accountID(), gurtd.resourcePrefix()))
}

func (gurtd *GenerateUserReportTaskDef) reportStatusTableName() *string {
	return aws.String(fmt.Sprintf("%s-user-report-status", gurtd.resourcePrefix()))
}

func (gurtd *GenerateUserReportTaskDef) containerDefinition() *ecs.ContainerDefinition {
	return &ecs.ContainerDefinition{
		Image:     gurtd.image(),
		Cpu:       aws.Int64(1024),
		Essential: aws.Bool(true),
		Memory:    aws.Int64(2048),
		Name:      aws.String(fmt.Sprintf("%s-generate-user-report", gurtd.resourcePrefix())),
		LogConfiguration: &ecs.LogConfiguration{
			LogDriver: aws.String("awslogs"),
			Options: map[string]*string{
				"awslogs-group":         aws.String(fmt.Sprintf("%s/gdpr", gurtd.resourcePrefix())),
				"awslogs-region":        aws.String(regionUsWest2),
				"awslogs-stream-prefix": aws.String("ecs-task-generate-user-report"),
			},
		},
		Environment: []*ecs.KeyValuePair{
			{
				Name:  aws.String("ENVIRONMENT"),
				Value: aws.String(gurtd.Environment),
			},
			{
				Name:  aws.String("REGION"),
				Value: aws.String(regionUsWest2),
			},
			{
				Name:  aws.String("QUEUE_URL"),
				Value: gurtd.queueURL(),
			},
			{
				Name:  aws.String("REPORT_STATUS_TABLE_NAME"),
				Value: gurtd.reportStatusTableName(),
			},
		},
	}
}

func (gurtd *GenerateUserReportTaskDef) taskDefinition() *ecs.RegisterTaskDefinitionInput {
	return &ecs.RegisterTaskDefinitionInput{
		ContainerDefinitions: []*ecs.ContainerDefinition{
			gurtd.containerDefinition(),
		},
		Cpu:              aws.String("1024"),
		Memory:           aws.String("2048"),
		Family:           aws.String(fmt.Sprintf("%s-generate-user-report", gurtd.resourcePrefix())),
		ExecutionRoleArn: gurtd.getExecuteRoleArn(),
		NetworkMode:      aws.String("awsvpc"),
		TaskRoleArn:      gurtd.getTaskRoleArn(),
	}
}

// Register register the task definition
func (gurtd *GenerateUserReportTaskDef) Register() (*ecs.RegisterTaskDefinitionOutput, error) {

	if err := gurtd.init(); err != nil {
		return nil, err
	}
	return gurtd.ecs.RegisterTaskDefinition(gurtd.taskDefinition())
}
