package gdpr

import (
	"context"
	"fmt"
	"net/http"
	"strings"
	"testing"
	"time"

	"code.justin.tv/foundation/history-service/internal/awsmocks"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/expression"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3manager"
	"github.com/sirupsen/logrus"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type GDPRSuite struct {
	suite.Suite
	client *Client

	mockedDDB *awsmocks.DynamoDBAPI
	mockedS3  *awsmocks.S3API
}

func (s *GDPRSuite) SetupTest() {
	s.mockedDDB = new(awsmocks.DynamoDBAPI)
	s.mockedS3 = new(awsmocks.S3API)

	uploader := s3manager.Uploader{
		S3:             s.mockedS3,
		MaxUploadParts: 1,
	}

	s.client = &Client{
		Environment: "testing",
		Logger:      logrus.New(),

		// s3 for presigner
		s3:                 s3.New(session.Must(session.NewSession(&aws.Config{Region: aws.String("us-west-2")}))),
		ddbClient:          s.mockedDDB,
		ddbAuditsTableName: "test-table",

		uploader:       uploader,
		uploadLocation: "s3-test-location",
	}

	s.client.initOnce.Do(func() {})
}

func (s *GDPRSuite) TearDownTest() {
	s.mockedDDB.AssertExpectations(s.T())
	s.mockedS3.AssertExpectations(s.T())
}

func (s *GDPRSuite) TestQueryDynamoDBNoResults() {
	expr := s.buildDynamodbExpression("123")

	handlerCalled := false
	handler := func(context.Context, *dynamodb.QueryOutput) error {
		handlerCalled = true
		return nil
	}

	queryInput := &dynamodb.QueryInput{
		TableName:                 aws.String(s.client.ddbAuditsTableName),
		ExpressionAttributeValues: expr.Values(),
		ExpressionAttributeNames:  expr.Names(),
		KeyConditionExpression:    expr.KeyCondition(),
		ProjectionExpression:      expr.Projection(),
	}

	ctx := context.Background()
	s.mockedDDB.On("QueryWithContext", ctx, queryInput).Return(&dynamodb.QueryOutput{}, nil)
	err := s.client.queryDynamodb(ctx, expr, handler)
	s.Require().NoError(err)
	s.Assert().True(handlerCalled)
}

func (s *GDPRSuite) TestQueryDynamoDBWithMultiPageResult() {
	expr := s.buildDynamodbExpression("123")

	handlerCallCount := 0
	handler := func(context.Context, *dynamodb.QueryOutput) error {
		handlerCallCount++
		return nil
	}

	queryInput := &dynamodb.QueryInput{
		TableName:                 aws.String(s.client.ddbAuditsTableName),
		ExpressionAttributeValues: expr.Values(),
		ExpressionAttributeNames:  expr.Names(),
		KeyConditionExpression:    expr.KeyCondition(),
		ProjectionExpression:      expr.Projection(),
	}
	ctx := context.Background()
	keyAtt := map[string]*dynamodb.AttributeValue{
		"user_id": {S: aws.String("123")},
	}

	s.mockedDDB.On("QueryWithContext", ctx, queryInput).Return(&dynamodb.QueryOutput{
		LastEvaluatedKey: keyAtt,
	}, nil)

	queryInput2 := &dynamodb.QueryInput{
		TableName:                 aws.String(s.client.ddbAuditsTableName),
		ExpressionAttributeValues: expr.Values(),
		ExpressionAttributeNames:  expr.Names(),
		KeyConditionExpression:    expr.KeyCondition(),
		ProjectionExpression:      expr.Projection(),
		ExclusiveStartKey:         keyAtt,
	}

	s.mockedDDB.On("QueryWithContext", ctx, queryInput2).Return(&dynamodb.QueryOutput{}, nil)
	err := s.client.queryDynamodb(context.Background(), expr, handler)
	s.Require().NoError(err)
	s.Assert().Equal(2, handlerCallCount)
}

func (s *GDPRSuite) buildDynamodbExpression(userID string) expression.Expression {
	keyCondition := expression.Key("user_id").Equal(expression.Value(userID))
	proj := expression.NamesList(
		expression.Name("user_id"),
		expression.Name("created_at"),
		expression.Name("action"),
		expression.Name("description"),
	)
	expr, err := expression.NewBuilder().
		WithKeyCondition(keyCondition).
		WithProjection(proj).
		Build()

	s.Require().NoError(err)
	return expr
}

type DeleteAuditsSuite struct {
	GDPRSuite
}

func (das *DeleteAuditsSuite) TestDeleteAuditsFromDynamoDB() {
	uid := "123"
	queryInput := &dynamodb.QueryInput{
		TableName: aws.String(das.client.ddbAuditsTableName),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":0": {
				S: aws.String(uid),
			},
		},
		ExpressionAttributeNames: map[string]*string{
			"#0": aws.String("user_id"),
			"#1": aws.String("action_created_at"),
		},
		KeyConditionExpression: aws.String("#0 = :0"),
		ProjectionExpression:   aws.String("#0, #1"),
	}

	ctx := context.Background()
	das.mockedDDB.On("QueryWithContext", ctx, queryInput).Return(&dynamodb.QueryOutput{}, nil)

	err := das.client.deleteAuditsFromDynamoDB(ctx, uid)
	das.Assert().NoError(err)
}

func (das *DeleteAuditsSuite) TestBatchDeleteItemsNoItems() {
	output := &dynamodb.QueryOutput{}
	err := das.client.batchDeleteItems(context.Background(), output)
	das.Assert().NoError(err)
}

func (das *DeleteAuditsSuite) TestBatchDeleteItemsOneDeleteBatch() {
	userID := "123"
	actionCreatedAt := fmt.Sprintf("fake-action-%d", time.Now().UnixNano())
	output := das.buildQueryOutputForBatchDelete(writeBatchSize, userID, actionCreatedAt)
	batchWriteItemInput := das.buildBatchWriteInput(writeBatchSize, userID, actionCreatedAt)

	ctx := context.Background()
	das.mockedDDB.On("BatchWriteItemWithContext", ctx, batchWriteItemInput).Return(&dynamodb.BatchWriteItemOutput{}, nil)
	err := das.client.batchDeleteItems(ctx, output)
	das.Assert().NoError(err)
}

func (das *DeleteAuditsSuite) TestBatchDeleteItemsOneDeleteBatchUnprocessedItem() {
	userID := "123"
	actionCreatedAt := fmt.Sprintf("fake-action-%d", time.Now().UnixNano())
	output := das.buildQueryOutputForBatchDelete(writeBatchSize, userID, actionCreatedAt)
	batchWriteItemInput := das.buildBatchWriteInput(writeBatchSize, userID, actionCreatedAt)

	ctx := context.Background()
	unprocessedItem := map[string][]*dynamodb.WriteRequest{
		das.client.ddbAuditsTableName: {
			{
				DeleteRequest: &dynamodb.DeleteRequest{
					Key: map[string]*dynamodb.AttributeValue{
						"user_id":           {S: aws.String(userID)},
						"action_created_at": {S: aws.String(actionCreatedAt)},
					},
				},
			},
		},
	}

	das.mockedDDB.On("BatchWriteItemWithContext", ctx, batchWriteItemInput).Return(&dynamodb.BatchWriteItemOutput{
		UnprocessedItems: unprocessedItem,
	}, nil)

	das.mockedDDB.On("BatchWriteItemWithContext", ctx, &dynamodb.BatchWriteItemInput{
		RequestItems: unprocessedItem,
	}).Return(&dynamodb.BatchWriteItemOutput{}, nil)

	err := das.client.batchDeleteItems(ctx, output)
	das.Assert().NoError(err)
}

func (das *DeleteAuditsSuite) TestBatchDeleteItemsMultipleDeleteBatch() {
	userID := "123"
	batchSize := 4*writeBatchSize + 10

	actionCreatedAt := fmt.Sprintf("fake-action-%d", time.Now().UnixNano())
	output := das.buildQueryOutputForBatchDelete(batchSize, userID, actionCreatedAt)

	batchWriteItemInput := das.buildBatchWriteInput(writeBatchSize, userID, actionCreatedAt)
	batchWriteItemInput2 := das.buildBatchWriteInput(10, userID, actionCreatedAt)

	ctx := context.Background()
	das.mockedDDB.On("BatchWriteItemWithContext", ctx, batchWriteItemInput).Return(&dynamodb.BatchWriteItemOutput{}, nil).Times(4)
	das.mockedDDB.On("BatchWriteItemWithContext", ctx, batchWriteItemInput2).Return(&dynamodb.BatchWriteItemOutput{}, nil)
	err := das.client.batchDeleteItems(ctx, output)
	das.Assert().NoError(err)
}

func (das *DeleteAuditsSuite) buildBatchWriteInput(itemCount int, userID, actionCreatedAt string) *dynamodb.BatchWriteItemInput {

	var writeRequests []*dynamodb.WriteRequest
	for i := 0; i < itemCount; i++ {
		wr := &dynamodb.WriteRequest{
			DeleteRequest: &dynamodb.DeleteRequest{
				Key: map[string]*dynamodb.AttributeValue{
					"user_id":           {S: aws.String(userID)},
					"action_created_at": {S: aws.String(actionCreatedAt)},
				},
			},
		}
		writeRequests = append(writeRequests, wr)
	}

	return &dynamodb.BatchWriteItemInput{
		RequestItems: map[string][]*dynamodb.WriteRequest{
			das.client.ddbAuditsTableName: writeRequests,
		},
	}
}

func (das *DeleteAuditsSuite) buildQueryOutputForBatchDelete(itemCount int, userID, actionCreatedAt string) *dynamodb.QueryOutput {

	var items []map[string]*dynamodb.AttributeValue

	for i := 0; i < itemCount; i++ {
		item := map[string]*dynamodb.AttributeValue{
			"user_id":           {S: aws.String(userID)},
			"action_created_at": {S: aws.String(actionCreatedAt)},
		}
		items = append(items, item)
	}
	return &dynamodb.QueryOutput{
		Count: aws.Int64(int64(len(items))),
		Items: items,
	}
}

type GenerateUserReportSuite struct {
	GDPRSuite
}

func (gurs *GenerateUserReportSuite) TestGenerateUserReportEmpty() {
	ctx := context.Background()

	gurs.mockedDDB.On("QueryWithContext", mock.Anything, mock.Anything).Return(&dynamodb.QueryOutput{}, nil).Once()

	fakeReq, err := http.NewRequest("GET", "http://example.com", nil)
	gurs.Require().NoError(err)

	gurs.mockedS3.On("PutObjectRequest", mock.Anything).Run(func(args mock.Arguments) {
		arg := args.Get(0).(*s3.PutObjectInput)
		gurs.Assert().Equal(gurs.client.uploadLocation, aws.StringValue(arg.Bucket))
		gurs.Assert().Contains(aws.StringValue(arg.Key), "report-123-")
		b := make([]byte, 1024)
		n, err := arg.Body.Read(b)
		gurs.Assert().NoError(err)
		content := string(b[0:n])
		gurs.Assert().Equal("UserID,Action,Description,Timestamp\n", content)
	}).Return(&request.Request{
		HTTPRequest: fakeReq,
	}, &s3.PutObjectOutput{}).Once()

	output, err := gurs.client.GenerateUserReport(ctx, "123")
	gurs.Require().NoError(err)
	gurs.Assert().Contains(output.Key, "report-123")
	gurs.Assert().Equal(output.Bucket, gurs.client.uploadLocation)
}

func (gurs *GenerateUserReportSuite) buildQueryOutput(itemCount int) *dynamodb.QueryOutput {

	var items []map[string]*dynamodb.AttributeValue

	for i := 0; i < itemCount; i++ {
		item := map[string]*dynamodb.AttributeValue{
			"user_id":     {S: aws.String("123")},
			"action":      {S: aws.String("testAction")},
			"description": {S: aws.String("something")},
			"created_at":  {N: aws.String("1234")},
		}
		items = append(items, item)
	}
	return &dynamodb.QueryOutput{
		Count: aws.Int64(int64(len(items))),
		Items: items,
	}
}

func (gurs *GenerateUserReportSuite) TestGenerateUserReportNotEmpty() {

	ctx := context.Background()
	itemCount := 1000
	qo := gurs.buildQueryOutput(itemCount)
	gurs.mockedDDB.On("QueryWithContext", mock.Anything, mock.Anything).Return(qo, nil).Once()

	fakeReq, err := http.NewRequest("GET", "http://example.com", nil)
	gurs.Require().NoError(err)

	var expectedContent []string
	expectedContent = append(expectedContent, "UserID,Action,Description,Timestamp\n")
	for i := 1; i < itemCount+1; i++ {
		expectedContent = append(expectedContent, "123,testAction,something,1234\n")
	}

	expContent := strings.Join(expectedContent, "")

	gurs.mockedS3.On("PutObjectRequest", mock.Anything).Run(func(args mock.Arguments) {
		arg := args.Get(0).(*s3.PutObjectInput)
		gurs.Assert().Equal(gurs.client.uploadLocation, aws.StringValue(arg.Bucket))
		gurs.Assert().Contains(aws.StringValue(arg.Key), "report-123-")
		b := make([]byte, (itemCount+1)*64)
		n, err := arg.Body.Read(b)
		gurs.Assert().NoError(err)
		content := string(b[0:n])
		gurs.Assert().Equal(expContent, content)
	}).Return(&request.Request{
		HTTPRequest: fakeReq,
	}, &s3.PutObjectOutput{})

	output, err := gurs.client.GenerateUserReport(ctx, "123")
	gurs.Require().NoError(err)
	gurs.Assert().Contains(output.Key, "report-123")
	gurs.Assert().Equal(output.Bucket, gurs.client.uploadLocation)
}

func TestGDPRTestSuite(t *testing.T) {
	suite.Run(t, new(GDPRSuite))
	suite.Run(t, new(DeleteAuditsSuite))
	suite.Run(t, new(GenerateUserReportSuite))
}
