package main

import (
	"context"
	"encoding/json"
	"errors"
	"testing"
	"time"

	"code.justin.tv/foundation/history-service/internal/awsmocks"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type HardDeleteEventHandlerSuite struct {
	suite.Suite
	h           *hardDeleteEventHandler
	mockedDDB   *awsmocks.DynamoDBAPI
	mockedQueue *awsmocks.SQSAPI
}

func (s *HardDeleteEventHandlerSuite) SetupTest() {
	s.mockedDDB = new(awsmocks.DynamoDBAPI)
	s.mockedQueue = new(awsmocks.SQSAPI)
	s.h = &hardDeleteEventHandler{
		ddb:       s.mockedDDB,
		queue:     s.mockedQueue,
		queueURL:  "fake-queue-url",
		tableName: "fake-table-name",
	}
}

func (s *HardDeleteEventHandlerSuite) TeardownTest() {
	s.mockedDDB.AssertExpectations(s.T())
	s.mockedQueue.AssertExpectations(s.T())
}

func (s *HardDeleteEventHandlerSuite) TestHandlerSuccess() {
	queueRH := "test"
	sqsEvent := events.SQSEvent{
		Records: []events.SQSMessage{
			s.buildSQSMessage("123", time.Now(), queueRH),
			s.buildSQSMessage("123", time.Now(), queueRH),
		},
	}

	ctx := context.Background()
	s.mockedDDB.On("PutItemWithContext", ctx, mock.Anything).Return(&dynamodb.PutItemOutput{}, nil).Times(2)
	s.mockedQueue.On("DeleteMessageWithContext", ctx, &sqs.DeleteMessageInput{
		ReceiptHandle: aws.String(queueRH),
		QueueUrl:      aws.String(s.h.queueURL),
	}).Return(&sqs.DeleteMessageOutput{}, nil).Times(2)

	err := s.h.handler(context.Background(), sqsEvent)
	s.Assert().NoError(err)
}

func (s *HardDeleteEventHandlerSuite) TestHandlerFailureDDBError() {
	sqsEvent := events.SQSEvent{
		Records: []events.SQSMessage{
			s.buildSQSMessage("123", time.Now(), "test"),
		},
	}

	ctx := context.Background()
	s.mockedDDB.On("PutItemWithContext", ctx, mock.Anything).Return(&dynamodb.PutItemOutput{}, errors.New("failed")).Once()
	err := s.h.handler(context.Background(), sqsEvent)
	s.Assert().Error(err)
}

func (s *HardDeleteEventHandlerSuite) TestHandlerFailureDeleteMessageFailed() {
	queueRH := "test"
	sqsEvent := events.SQSEvent{
		Records: []events.SQSMessage{
			s.buildSQSMessage("123", time.Now(), queueRH),
			s.buildSQSMessage("123", time.Now(), queueRH),
		},
	}

	ctx := context.Background()
	s.mockedDDB.On("PutItemWithContext", ctx, mock.Anything).Return(&dynamodb.PutItemOutput{}, nil).Once()
	s.mockedQueue.On("DeleteMessageWithContext", ctx, &sqs.DeleteMessageInput{
		ReceiptHandle: aws.String(queueRH),
		QueueUrl:      aws.String(s.h.queueURL),
	}).Return(nil, errors.New("delete message failed")).Once()

	err := s.h.handler(context.Background(), sqsEvent)
	s.Assert().Error(err)
}

func (s *HardDeleteEventHandlerSuite) buildSQSMessage(userID string, ts time.Time, rh string) events.SQSMessage {
	msg := &SNSHardDeleteEvent{
		UserID:    userID,
		Timestamp: ts,
	}
	m, err := json.Marshal(msg)
	s.Require().NoError(err)

	return events.SQSMessage{
		ReceiptHandle: rh,
		Body:          string(m),
	}
}

func TestDeleteNotificationHandler(t *testing.T) {
	suite.Run(t, new(HardDeleteEventHandlerSuite))
}
