package main

import (
	"context"
	"encoding/json"
	"fmt"
	"log"
	"os"
	"time"

	"code.justin.tv/foundation/history-service/internal/historyin"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-lambda-go/lambda"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
)

// BatchWriteChunkSize BatchWriteItemChunkSize
const BatchWriteChunkSize = 25

type dynamodbHistoryAudit struct {
	UserID          string `dynamodbav:"user_id"`
	Action          string `dynamodbav:"action"`
	ActionCreatedAt string `dynamodbav:"action_created_at"`

	UserType     string                `dynamodbav:"user_type,omitempty"`
	ResourceType string                `dynamodbav:"resource_type,omitempty"`
	ResourceID   string                `dynamodbav:"resource_id,omitempty"`
	Description  string                `dynamodbav:"description,omitempty"`
	Changes      []historyin.ChangeSet `dynamodbav:"changes,omitempty"`

	// Optional Attributes
	UUID      historyin.UUID     `dynamodbav:"uuid,omitempty"`
	CreatedAt historyin.Time     `dynamodbav:"created_at"`
	TTL       historyin.Duration `dynamodbav:"ttl,omitempty"`
	Source    string             `dynamodbav:"source,omitempty"`
}

func toDynamoDBHistoryAudit(audit historyin.Audit) dynamodbHistoryAudit {
	source := os.Getenv("EVENT_SOURCE")
	if source == "" {
		source = "history-service"
	}

	return dynamodbHistoryAudit{
		Action:          audit.Action,
		UserType:        audit.UserType,
		UserID:          audit.UserID,
		ResourceType:    audit.ResourceType,
		ResourceID:      audit.ResourceID,
		Description:     audit.Description,
		Changes:         audit.Changes,
		UUID:            audit.UUID,
		CreatedAt:       audit.CreatedAt,
		TTL:             audit.TTL,
		ActionCreatedAt: fmt.Sprintf("%s_%d", audit.Action, time.Time(audit.CreatedAt).UnixNano()),
		Source:          source,
	}
}

func buildBatchWriteItemInput(dynamoAudits []dynamodbHistoryAudit) (input *dynamodb.BatchWriteItemInput, err error) {
	var wr []*dynamodb.WriteRequest
	for _, audit := range dynamoAudits {
		var av map[string]*dynamodb.AttributeValue
		av, err = dynamodbattribute.MarshalMap(audit)
		if err != nil {
			return
		}
		req := &dynamodb.WriteRequest{
			PutRequest: &dynamodb.PutRequest{
				Item: av,
			},
		}
		wr = append(wr, req)
	}

	tableName := os.Getenv("DYNAMODB_TABLE_NAME")
	input = &dynamodb.BatchWriteItemInput{
		RequestItems: map[string][]*dynamodb.WriteRequest{
			tableName: wr,
		},
	}
	return
}

func batchWriteItem(ctx context.Context, ddb dynamodbiface.DynamoDBAPI, input *dynamodb.BatchWriteItemInput) (output *dynamodb.BatchWriteItemOutput, err error) {
	output, err = ddb.BatchWriteItemWithContext(ctx, input)
	if err != nil {
		log.Printf("failed to update dynamodb: %s", err.Error())
		return
	}

	// There are unproccessed items
	if len(output.UnprocessedItems) != 0 {
		output, err = batchWriteItem(ctx, ddb, &dynamodb.BatchWriteItemInput{
			RequestItems: output.UnprocessedItems,
		})
	}
	return
}

func splitRecordsInChunks(audits []dynamodbHistoryAudit) [][]dynamodbHistoryAudit {
	var chunks [][]dynamodbHistoryAudit
	totalRecords := len(audits)
	for i := 0; i < len(audits); i += BatchWriteChunkSize {
		end := i + BatchWriteChunkSize
		// end  - 1 is the index of last item to include in batch
		if end > totalRecords {
			end = totalRecords
		}
		chunks = append(chunks, audits[i:end])
	}
	return chunks
}

func dedupRecords(orig []dynamodbHistoryAudit) []dynamodbHistoryAudit {
	var deduped []dynamodbHistoryAudit
	keys := make(map[string]struct{})
	for _, record := range orig {
		key := fmt.Sprintf("%s_%s", record.UserID, record.ActionCreatedAt)
		if _, ok := keys[key]; !ok {
			keys[key] = struct{}{}
			deduped = append(deduped, record)
		}
	}
	return deduped
}

func processEvents(ctx context.Context, ddb dynamodbiface.DynamoDBAPI, kinesisEvent events.KinesisEvent) error {
	var dynamoAudits []dynamodbHistoryAudit

	for _, record := range kinesisEvent.Records {
		var historyV2Audit historyin.Audit
		kinesisRecord := record.Kinesis
		data := kinesisRecord.Data
		if len(data) == 0 {
			log.Printf("records with no data present, continue...")
			continue
		}
		if err := json.Unmarshal(data, &historyV2Audit); err != nil {
			return err
		}
		dynamoAudits = append(dynamoAudits, toDynamoDBHistoryAudit(historyV2Audit))
	}

	// Dedup records .
	dynamoAudits = dedupRecords(dynamoAudits)
	//Chunk the records into batch size for BatchWriteInput
	chunks := splitRecordsInChunks(dynamoAudits)

	log.Printf("%d records to process", len(dynamoAudits))
	for _, chunk := range chunks {

		input, err := buildBatchWriteItemInput(chunk)
		if err != nil {
			return err
		}

		_, err = batchWriteItem(ctx, ddb, input)
		if err != nil {
			return err
		}
	}
	log.Printf("Done.")
	return nil
}

func handler(ctx context.Context, kinesisEvent events.KinesisEvent) error {
	sess, err := session.NewSession()
	if err != nil {
		return err
	}
	return processEvents(ctx, dynamodb.New(sess), kinesisEvent)
}

func main() {
	lambda.Start(handler)
}
