package main

import (
	"context"
	"encoding/json"
	"testing"
	"time"

	"code.justin.tv/foundation/history-service/internal/awsmocks"
	"code.justin.tv/foundation/history-service/internal/historyin"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
)

func buildKinesisEvent(data []byte) events.KinesisEvent {

	var kinesisRecord events.KinesisRecord
	if len(data) > 0 {
		kinesisRecord = events.KinesisRecord{
			Data:                        data,
			EncryptionType:              "encryption-type",
			PartitionKey:                "partition-key",
			SequenceNumber:              "seq-number",
			KinesisSchemaVersion:        "kinesis-schema-version",
			ApproximateArrivalTimestamp: events.SecondsEpochTime{Time: time.Now()},
		}
	}
	return events.KinesisEvent{
		Records: []events.KinesisEventRecord{
			{
				EventName:         "stream-event-name",
				EventSource:       "stream-source",
				EventSourceArn:    "event-source-arn",
				EventVersion:      "event-version-1",
				InvokeIdentityArn: "identity-arn",
				Kinesis:           kinesisRecord,
			},
		},
	}
}

func historyAudit() historyin.Audit {
	return buildHistoryAudit("1234", "some_random_action", historyin.Time(time.Now()))
}

func buildHistoryAudit(userID, action string, createdAt historyin.Time) historyin.Audit {
	return historyin.Audit{
		Action:       action,
		UserType:     "twitch-user:12344",
		UserID:       userID,
		ResourceType: "type",
		Description:  "test-description",
		CreatedAt:    createdAt,
		UUID:         "abcdefabcdefabcdefabcdefabcdefabcdef",
		Changes: []historyin.ChangeSet{
			{
				Attribute: "user",
				OldValue:  "old",
				NewValue:  "new",
			},
			{
				Attribute: "password",
				OldValue:  "12245",
				NewValue:  "something",
			},
		},
	}
}

func historyEventAsByte() (event []byte) {
	audit := historyAudit()
	event, _ = json.Marshal(audit)
	return
}

func TestToDynamoHistoryAudit(t *testing.T) {
	t.Run("toDynamoDBHistoryAudit", func(t *testing.T) {
		ddbAudit := toDynamoDBHistoryAudit(historyAudit())
		require.NotNil(t, ddbAudit.ActionCreatedAt)

		t.Run("is marshalled correctly", func(t *testing.T) {
			out, err := dynamodbattribute.MarshalMap(ddbAudit)
			require.NoError(t, err)
			t.Run("Changes, and UUID are set correctly", func(t *testing.T) {
				audit := &dynamodbHistoryAudit{}
				err := dynamodbattribute.UnmarshalMap(out, audit)
				assert.NoError(t, err)
				assert.Equal(t, ddbAudit.UUID, audit.UUID)
				assert.Equal(t, ddbAudit.Changes, audit.Changes)
			})
		})

	})

}

func TestHandler(t *testing.T) {
	t.Run("processEvents", func(t *testing.T) {
		mockedDDB := &awsmocks.DynamoDBAPI{}
		ctx := context.Background()
		t.Run("No records in steam", func(t *testing.T) {
			kinesisEvent := buildKinesisEvent([]byte(""))
			err := processEvents(ctx, mockedDDB, kinesisEvent)
			assert.NoError(t, err)
		})

		t.Run("Records were processed", func(t *testing.T) {
			mockedDDB.On("BatchWriteItemWithContext", mock.Anything, mock.Anything, mock.Anything).Return(&dynamodb.BatchWriteItemOutput{}, nil).Once()
			kinesisEvent := buildKinesisEvent(historyEventAsByte())
			err := processEvents(ctx, mockedDDB, kinesisEvent)
			assert.NoError(t, err)
		})

		t.Run("with unproccessed records", func(t *testing.T) {
			unprocessedEvent := &dynamodb.BatchWriteItemOutput{
				UnprocessedItems: map[string][]*dynamodb.WriteRequest{
					"table_name": {
						&dynamodb.WriteRequest{
							PutRequest: &dynamodb.PutRequest{},
						},
					},
				},
			}

			mockedDDB.On("BatchWriteItemWithContext", mock.Anything, mock.Anything).Return(unprocessedEvent, nil).Once()
			mockedDDB.On("BatchWriteItemWithContext", mock.Anything, mock.Anything).Return(&dynamodb.BatchWriteItemOutput{}, nil).Once()
			kinesisEvent := buildKinesisEvent(historyEventAsByte())
			err := processEvents(ctx, mockedDDB, kinesisEvent)
			assert.NoError(t, err)
		})
	})

	t.Run("splitRecordsInChunks", func(t *testing.T) {
		t.Run("No records", func(t *testing.T) {
			chunks := splitRecordsInChunks([]dynamodbHistoryAudit{})
			assert.Equal(t, len(chunks), 0)
		})
		t.Run("same as BatchWriteChunkSize records", func(t *testing.T) {
			var records []dynamodbHistoryAudit
			for i := 0; i < BatchWriteChunkSize; i++ {
				records = append(records, toDynamoDBHistoryAudit(historyAudit()))
			}
			chunks := splitRecordsInChunks(records)
			require.Equal(t, len(chunks), 1)
			assert.Equal(t, len(chunks[0]), BatchWriteChunkSize)
		})
		t.Run("more records than BatchWriteChunkSize", func(t *testing.T) {
			var records []dynamodbHistoryAudit
			for i := 0; i < 4*BatchWriteChunkSize+1; i++ {
				records = append(records, toDynamoDBHistoryAudit(historyAudit()))
			}

			chunks := splitRecordsInChunks(records)
			require.Equal(t, len(chunks), 5)
			for i := 0; i < 4; i++ {

				assert.Equal(t, len(chunks[i]), BatchWriteChunkSize)
			}
			assert.Equal(t, len(chunks[4]), 1)

		})
	})

	t.Run("dedupRecords", func(t *testing.T) {
		t.Run("No records", func(t *testing.T) {
			chunks := dedupRecords([]dynamodbHistoryAudit{})
			assert.Equal(t, len(chunks), 0)
		})

		t.Run("depulicate records", func(t *testing.T) {
			var rec []dynamodbHistoryAudit
			audit := toDynamoDBHistoryAudit(historyAudit())
			rec = append(rec, audit, audit)
			rec = dedupRecords(rec)
			require.Equal(t, len(rec), 1)
			assert.Equal(t, rec[0], audit)
		})

		t.Run("depulicate records", func(t *testing.T) {
			var rec []dynamodbHistoryAudit

			now := time.Now()
			historyTime := historyin.Time(now)
			oldTime := historyin.Time(now.Add(time.Hour * 24 * 365))

			rec = append(rec,
				toDynamoDBHistoryAudit(buildHistoryAudit("someUser", "random-action", historyTime)),
				toDynamoDBHistoryAudit(buildHistoryAudit("someUser", "random-action", historyTime)), //duplicate record
				toDynamoDBHistoryAudit(buildHistoryAudit("someUser", "random-action", oldTime)),
				toDynamoDBHistoryAudit(buildHistoryAudit("someUser", "other-action", historyTime)),
				toDynamoDBHistoryAudit(buildHistoryAudit("someUser", "other-action", oldTime)),
				toDynamoDBHistoryAudit(buildHistoryAudit("anotherUser", "random-action", oldTime)),
			)
			rec = dedupRecords(rec)
			require.Equal(t, len(rec), 5)
		})
	})
}
