data "aws_iam_policy_document" "task-arp" {
  statement {
    actions = [
      "sts:AssumeRole",
    ]

    principals {
      type        = "Service"
      identifiers = ["ecs-tasks.amazonaws.com"]
    }
  }
}

resource "aws_iam_role" "task-delete-user" {
  name               = "${local.resource_prefix}-task-delete-user"
  assume_role_policy = "${data.aws_iam_policy_document.task-arp.json}"
}

resource "aws_iam_role" "task-generate-user-report" {
  name               = "${local.resource_prefix}-task-generate-user-report"
  assume_role_policy = "${data.aws_iam_policy_document.task-arp.json}"
}

data "aws_iam_policy_document" "task-delete-user-p" {
  statement {
    actions = [
      "dynamodb:Query",
      "dynamodb:BatchWriteItem",
    ]

    effect = "Allow"

    resources = [
      "${var.user_audits_table_arn}",
    ]
  }

  statement {
    actions = [
      "sqs:DeleteMessage",
      "sqs:ReceiveMessage",
    ]

    effect = "Allow"

    resources = [
      "${aws_sqs_queue.delete-user.arn}",
    ]
  }

  statement {
    actions = [
      "es:ESHttpDelete",
      "es:ESHttpPost",
      "es:ESHttpHead",
    ]

    effect = "Allow"

    resources = [
      "${var.elasticsearch_domain_arn}/*",
    ]
  }

  statement {
    actions = [
      "kms:Decrypt",
      "kms:DescribeKey",
    ]

    effect = "Allow"

    resources = [
      "${aws_kms_key.sqs.arn}",
    ]
  }
}

resource "aws_iam_role_policy" "task-delete-user" {
  role   = "${aws_iam_role.task-delete-user.id}"
  policy = "${data.aws_iam_policy_document.task-delete-user-p.json}"
}

data "aws_iam_policy_document" "task-generate-user-report-p" {
  statement {
    actions = [
      "s3:*",
    ]

    resources = [
      "${aws_s3_bucket.user-reports.arn}/*",
    ]
  }

  statement {
    actions = [
      "dynamodb:Query",
    ]

    effect = "Allow"

    resources = [
      "${var.user_audits_table_arn}",
    ]
  }

  statement {
    actions = [
      "dynamodb:Query",
      "dynamodb:UpdateItem",
    ]

    effect = "Allow"

    resources = [
      "${aws_dynamodb_table.report-status.arn}",
    ]
  }

  statement {
    actions = [
      "sqs:ReceiveMessage",
      "sqs:DeleteMessage",
    ]

    effect = "Allow"

    resources = [
      "${aws_sqs_queue.generate-user-report.arn}",
    ]
  }

  statement {
    actions = [
      "kms:Decrypt",
      "kms:DescribeKey",
    ]

    effect = "Allow"

    resources = [
      "${aws_kms_key.sqs.arn}",
    ]
  }
}

resource "aws_iam_role_policy" "task-generate-user-report" {
  role   = "${aws_iam_role.task-generate-user-report.id}"
  policy = "${data.aws_iam_policy_document.task-generate-user-report-p.json}"
}

# Execution IAM role for ECS Container
data "aws_iam_policy_document" "execute-arp" {
  statement {
    actions = [
      "sts:AssumeRole",
    ]

    principals {
      type        = "Service"
      identifiers = ["ecs-tasks.amazonaws.com", "ecs.amazonaws.com"]
    }
  }
}

resource "aws_iam_role" "execute-delete-user" {
  name               = "${local.resource_prefix}-execute-delete-user"
  assume_role_policy = "${data.aws_iam_policy_document.execute-arp.json}"
}

resource "aws_iam_role_policy_attachment" "execute-delete-user" {
  role       = "${aws_iam_role.execute-delete-user.name}"
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonECSTaskExecutionRolePolicy"
}

resource "aws_iam_role" "execute-generate-user-report" {
  name               = "${local.resource_prefix}-execute-generate-user-report"
  assume_role_policy = "${data.aws_iam_policy_document.execute-arp.json}"
}

resource "aws_iam_role_policy_attachment" "execute-generate-user-report" {
  role       = "${aws_iam_role.execute-generate-user-report.name}"
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonECSTaskExecutionRolePolicy"
}

# lambda role #
data "aws_iam_policy_document" "lambda-arp" {
  statement {
    actions = [
      "sts:AssumeRole",
    ]

    principals {
      type        = "Service"
      identifiers = ["lambda.amazonaws.com"]
    }

    effect = "Allow"
  }
}

data "aws_iam_policy_document" "lambda-delete-user-p" {
  statement {
    effect = "Allow"

    actions = [
      "dynamodb:DescribeStream",
      "dynamodb:GetRecords",
      "dynamodb:GetShardIterator",
      "dynamodb:ListStreams",
    ]

    resources = [
      "${aws_dynamodb_table.pending-user-deletions.arn}/stream/*",
    ]
  }

  statement {
    actions = [
      "sqs:SendMessage",
    ]

    effect = "Allow"

    resources = [
      "${aws_sqs_queue.delete-user.arn}",
    ]
  }

  statement {
    actions = [
      "ecs:RunTask",
    ]

    effect = "Allow"

    resources = [
      "arn:aws:ecs:us-west-2:${local.aws_account_id}:task-definition/${local.resource_prefix}-delete-user",
    ]
  }

  statement {
    actions = [
      "kms:GenerateDataKey*",
      "kms:DescribeKey",
      "kms:Encrypt",
      "kms:Decrypt",
    ]

    effect = "Allow"

    resources = [
      "${aws_kms_key.sqs.arn}",
    ]
  }

  statement {
    actions = [
      "iam:PassRole",
    ]

    resources = [
      "${aws_iam_role.execute-delete-user.arn}",
      "${aws_iam_role.task-delete-user.arn}",
    ]
  }

  statement {
    actions = [
      "logs:CreateLogGroup",
      "logs:CreateLogStream",
      "logs:PutLogEvents",
    ]

    resources = ["*"]
    effect    = "Allow"
  }
}

resource "aws_iam_role" "lambda-delete-user" {
  name               = "${local.resource_prefix}-lambda-delete-user"
  assume_role_policy = "${data.aws_iam_policy_document.lambda-arp.json}"
}

resource "aws_iam_role_policy" "lambda-delete-user" {
  name = "${local.resource_prefix}-lambda-delete-user-policy"
  role = "${aws_iam_role.lambda-delete-user.id}"

  policy = "${data.aws_iam_policy_document.lambda-delete-user-p.json}"
}

resource "aws_iam_role_policy_attachment" "lambda-delete-user-vpc" {
  policy_arn = "arn:aws:iam::aws:policy/service-role/AWSLambdaVPCAccessExecutionRole"
  role       = "${aws_iam_role.lambda-delete-user.name}"
}

data "aws_iam_policy_document" "lambda-generate-user-report-p" {
  statement {
    effect = "Allow"

    actions = [
      "dynamodb:DescribeStream",
      "dynamodb:GetRecords",
      "dynamodb:GetShardIterator",
      "dynamodb:ListStreams",
    ]

    resources = [
      "${aws_dynamodb_table.report-status.arn}/stream/*",
    ]
  }

  statement {
    actions = [
      "sqs:SendMessage",
    ]

    effect = "Allow"

    resources = [
      "${aws_sqs_queue.generate-user-report.arn}",
    ]
  }

  statement {
    actions = [
      "ecs:RunTask",
    ]

    effect = "Allow"

    resources = [
      "arn:aws:ecs:us-west-2:${local.aws_account_id}:task-definition/${local.resource_prefix}-generate-user-report",
    ]
  }

  statement {
    actions = [
      "kms:GenerateDataKey*",
      "kms:DescribeKey",
      "kms:Encrypt",
      "kms:Decrypt",
    ]

    effect = "Allow"

    resources = [
      "${aws_kms_key.sqs.arn}",
    ]
  }

  statement {
    actions = [
      "iam:PassRole",
    ]

    resources = [
      "${aws_iam_role.execute-generate-user-report.arn}",
      "${aws_iam_role.task-generate-user-report.arn}",
    ]
  }

  statement {
    actions = [
      "logs:CreateLogGroup",
      "logs:CreateLogStream",
      "logs:PutLogEvents",
    ]

    resources = ["*"]
    effect    = "Allow"
  }
}

resource "aws_iam_role" "lambda-generate-user-report" {
  name               = "${local.resource_prefix}-lambda-generate-user-report"
  assume_role_policy = "${data.aws_iam_policy_document.lambda-arp.json}"
}

resource "aws_iam_role_policy" "lambda-generate-user-report" {
  name = "${local.resource_prefix}-lambda-generate-user-report-policy"
  role = "${aws_iam_role.lambda-generate-user-report.id}"

  policy = "${data.aws_iam_policy_document.lambda-generate-user-report-p.json}"
}

resource "aws_iam_role_policy_attachment" "lambda-generate-user-report-vpc" {
  policy_arn = "arn:aws:iam::aws:policy/service-role/AWSLambdaVPCAccessExecutionRole"
  role       = "${aws_iam_role.lambda-generate-user-report.name}"
}

data "aws_iam_policy_document" "lambda-hard-delete-event-handler" {
  statement {
    effect = "Allow"

    actions = [
      "sqs:ReceiveMessage",
      "sqs:DeleteMessage",
      "sqs:GetQueueAttributes",
    ]

    resources = [
      "${aws_sqs_queue.user-service-hard-delete.arn}",
    ]
  }

  statement {
    effect = "Allow"

    actions = [
      "dynamodb:PutItem",
    ]

    resources = [
      "${aws_dynamodb_table.pending-user-deletions.arn}",
    ]
  }

  statement {
    actions = [
      "kms:GenerateDataKey*",
      "kms:DescribeKey",
      "kms:Decrypt",
    ]

    effect = "Allow"

    resources = [
      "${aws_kms_key.sqs.arn}",
    ]
  }

  statement {
    actions = [
      "logs:CreateLogGroup",
      "logs:CreateLogStream",
      "logs:PutLogEvents",
    ]

    resources = ["*"]
    effect    = "Allow"
  }
}

resource "aws_iam_role" "lambda-hard-delete-event-handler" {
  name               = "${local.resource_prefix}-lambda-hard-delete-event-handler"
  assume_role_policy = "${data.aws_iam_policy_document.lambda-arp.json}"
}

resource "aws_iam_role_policy" "lambda-hard-delete-event-handler" {
  name   = "${local.resource_prefix}-lambda-hard-delete-event-handler-policy"
  role   = "${aws_iam_role.lambda-hard-delete-event-handler.id}"
  policy = "${data.aws_iam_policy_document.lambda-hard-delete-event-handler.json}"
}

resource "aws_iam_role_policy_attachment" "lambda-hard-delete-event-handler-vpc" {
  policy_arn = "arn:aws:iam::aws:policy/service-role/AWSLambdaVPCAccessExecutionRole"
  role       = "${aws_iam_role.lambda-hard-delete-event-handler.name}"
}

data "aws_iam_policy_document" "service-arp" {
  statement {
    actions = [
      "sts:AssumeRole",
    ]

    principals {
      type = "AWS"

      identifiers = [
        "${var.gdpr_service_allowed_arn}",
        "${local.root_arn}",
      ]
    }
  }
}

data "aws_iam_policy_document" "service-p" {
  statement {
    actions = [
      "dynamodb:Query",
      "dynamodb:Scan",
      "dynamodb:PutItem",
    ]

    effect = "Allow"

    resources = [
      "${aws_dynamodb_table.report-status.arn}",
      "${aws_dynamodb_table.pending-user-deletions.arn}",
      "${aws_dynamodb_table.report-status.arn}/index/*",
      "${aws_dynamodb_table.pending-user-deletions.arn}/index/*",
    ]
  }

  statement {
    actions = [
      "es:ESHttpHead",
      "es:ESHttpGet",
      "es:ESHttpPost",
    ]

    effect = "Allow"

    resources = [
      "${var.elasticsearch_domain_arn}/*",
    ]
  }

  statement {
    actions = [
      "s3:GetObject",
    ]

    effect = "Allow"

    resources = [
      "${aws_s3_bucket.user-reports.arn}/*",
    ]
  }
}

resource "aws_iam_role" "service" {
  name               = "${local.resource_prefix}-gdpr-service"
  assume_role_policy = "${data.aws_iam_policy_document.service-arp.json}"
}

resource "aws_iam_role_policy" "service" {
  name   = "${local.resource_prefix}-gdpr-service"
  role   = "${aws_iam_role.service.id}"
  policy = "${data.aws_iam_policy_document.service-p.json}"
}
