resource "aws_dynamodb_table" "history" {
  name           = "${local.prefix}-audits"
  read_capacity  = "${local.read_capacity}"
  write_capacity = "${local.write_capacity}"

  hash_key  = "user_id"
  range_key = "action_created_at"

  attribute {
    name = "user_id"
    type = "S"
  }

  attribute {
    name = "resource_id"
    type = "S"
  }

  attribute {
    name = "resource_type"
    type = "S"
  }

  attribute {
    name = "action_created_at"
    type = "S"
  }

  attribute {
    name = "created_at"
    type = "N"
  }

  lifecycle {
    ignore_changes = [
      "read_capacity",
      "write_capacity",
    ]
  }

  point_in_time_recovery {
    enabled = true
  }

  global_secondary_index {
    name            = "user_id_created_at"
    hash_key        = "user_id"
    range_key       = "created_at"
    write_capacity  = "${local.write_capacity}"
    read_capacity   = "${local.read_capacity}"
    projection_type = "ALL"
  }

  global_secondary_index {
    name            = "resource_id_resource_type"
    hash_key        = "resource_id"
    range_key       = "resource_type"
    write_capacity  = "${local.write_capacity}"
    read_capacity   = "${local.read_capacity}"
    projection_type = "ALL"
  }
}

data "aws_iam_role" "autoscaling" {
  name = "AWSServiceRoleForApplicationAutoScaling_DynamoDBTable"
}

resource "aws_appautoscaling_target" "dynamodb_table_write_target" {
  min_capacity       = "${var.table_history_write_min_capacity}"
  max_capacity       = "${var.table_history_write_max_capacity}"
  resource_id        = "table/${aws_dynamodb_table.history.name}"
  role_arn           = "${data.aws_iam_role.autoscaling.arn}"
  scalable_dimension = "dynamodb:table:WriteCapacityUnits"
  service_namespace  = "dynamodb"

  lifecycle {
    ignore_changes = [
      "min_capacity",
      "max_capacity",
    ]
  }
}

resource "aws_appautoscaling_target" "dynamodb_index_write_target" {
  min_capacity       = "${var.table_history_write_min_capacity}"
  max_capacity       = "${var.table_history_write_max_capacity}"
  resource_id        = "table/${aws_dynamodb_table.history.name}/index/user_id_created_at"
  role_arn           = "${data.aws_iam_role.autoscaling.arn}"
  scalable_dimension = "dynamodb:index:WriteCapacityUnits"
  service_namespace  = "dynamodb"

  lifecycle {
    ignore_changes = [
      "min_capacity",
      "max_capacity",
    ]
  }
}

resource "aws_appautoscaling_policy" "dynamodb_table_write_policy" {
  name = "DynamoDBWriteCapacityUtilization:${aws_appautoscaling_target.dynamodb_table_write_target.resource_id}"

  policy_type        = "TargetTrackingScaling"
  resource_id        = "${aws_appautoscaling_target.dynamodb_table_write_target.resource_id}"
  scalable_dimension = "${aws_appautoscaling_target.dynamodb_table_write_target.scalable_dimension}"
  service_namespace  = "${aws_appautoscaling_target.dynamodb_table_write_target.service_namespace}"

  target_tracking_scaling_policy_configuration {
    predefined_metric_specification {
      predefined_metric_type = "DynamoDBWriteCapacityUtilization"
    }

    target_value = "${local.table_write_target_capacity_utilization}"
  }
}

resource "aws_appautoscaling_policy" "dynamodb_index_write_policy" {
  name = "DynamoDBWriteCapacityUtilization:${aws_appautoscaling_target.dynamodb_index_write_target.resource_id}"

  policy_type        = "TargetTrackingScaling"
  resource_id        = "${aws_appautoscaling_target.dynamodb_index_write_target.resource_id}"
  scalable_dimension = "${aws_appautoscaling_target.dynamodb_index_write_target.scalable_dimension}"
  service_namespace  = "${aws_appautoscaling_target.dynamodb_index_write_target.service_namespace}"

  target_tracking_scaling_policy_configuration {
    predefined_metric_specification {
      predefined_metric_type = "DynamoDBWriteCapacityUtilization"
    }

    target_value = "${local.table_write_target_capacity_utilization}"
  }
}
