resource "aws_elasticsearch_domain" "app" {
  domain_name           = "${local.prefix}"
  elasticsearch_version = "5.5"

  ebs_options {
    ebs_enabled = true
    volume_size = "${var.es_volume_size}"
    volume_type = "gp2"
  }

  cluster_config {
    instance_type            = "${var.es_instance_type}"
    instance_count           = "${var.es_instance_count}"
    dedicated_master_enabled = true
    dedicated_master_count   = "${var.es_master_instance_count}"
    dedicated_master_type    = "${var.es_master_instance_type}"
    zone_awareness_enabled   = "${var.es_zone_awareness_enabled}"
  }

  lifecycle {
    ignore_changes = [
      // TODO: this is apply stack as is since IpAddress in the condition
      // statement is different than what is in this file.
      "access_policies",

      "cluster_config.0.dedicated_master_type",
      "cluster_config.0.instance_type",
    ]
  }
}

resource "aws_elasticsearch_domain_policy" "app" {
  domain_name = "${aws_elasticsearch_domain.app.domain_name}"

  access_policies = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Principal": {
        "AWS": [
          "${local.root_arn}"
        ]
      },
      "Action": "es:*",
      "Resource": "${aws_elasticsearch_domain.app.arn}/*"
    },
    {
      "Effect": "Allow",
      "Principal": {
        "AWS": "*"
      },
      "Action": "es:*",
      "Resource": "${aws_elasticsearch_domain.app.arn}/*",
      "Condition": {
        "IpAddress": {
          "aws:SourceIp": [
            "35.164.244.5",
            "35.165.151.215",
            "35.165.152.80",
            "38.99.7.224/27",
            "38.99.10.96/29",
            "38.104.129.210"
          ]
        }
      }
    }
  ]
}
EOF
}

resource "aws_iam_role" "firehose-elasticsearch" {
  name               = "${local.prefix}-firehose-elasticsearch"
  assume_role_policy = "${data.aws_iam_policy_document.firehose-arp.json}"
}

data "aws_iam_policy_document" "firehose-arp" {
  statement {
    actions = [
      "sts:AssumeRole",
    ]

    principals {
      type = "Service"

      identifiers = [
        "firehose.amazonaws.com",
      ]
    }

    condition {
      test     = "StringEquals"
      variable = "sts:ExternalId"
      values   = ["${var.aws_account_id}"]
    }

    effect = "Allow"
  }
}

resource "aws_iam_role_policy" "firehose-elasticsearch" {
  role   = "${aws_iam_role.firehose-elasticsearch.id}"
  policy = "${data.aws_iam_policy_document.firehose-elasticsearch-rp.json}"
}

data "aws_iam_policy_document" "firehose-elasticsearch-rp" {
  statement {
    actions = [
      "logs:CreateLogStream",
      "logs:PutLogEvents",
    ]

    resources = [
      "${aws_cloudwatch_log_stream.elasticsearch.arn}",
    ]

    effect = "Allow"
  }

  statement {
    actions = [
      "es:DescribeElasticsearchDomain",
      "es:DescribeElasticsearchDomains",
      "es:DescribeElasticsearchDomainConfig",
      "es:ESHttpPost",
      "es:ESHttpPut",
    ]

    resources = [
      "${aws_elasticsearch_domain.app.arn}",
      "${aws_elasticsearch_domain.app.arn}/*",
    ]

    effect = "Allow"
  }

  statement {
    actions = [
      "es:ESHttpGet",
    ]

    resources = [
      "${aws_elasticsearch_domain.app.arn}/_all/_settings",
      "${aws_elasticsearch_domain.app.arn}/_cluster/stats",
      "${aws_elasticsearch_domain.app.arn}/history*/_mapping/type-name",
      "${aws_elasticsearch_domain.app.arn}/_nodes",
      "${aws_elasticsearch_domain.app.arn}/_nodes/stats",
      "${aws_elasticsearch_domain.app.arn}/_nodes/*/stats",
      "${aws_elasticsearch_domain.app.arn}/_stats",
      "${aws_elasticsearch_domain.app.arn}/history*/_stats",
    ]

    effect = "Allow"
  }
}
