module "dummy_go_lambda" {
  source = "../dummy-go-lambda"
}

data "aws_iam_policy" "kinesis-execution" {
  arn = "arn:aws:iam::aws:policy/service-role/AWSLambdaKinesisExecutionRole"
}

data "aws_iam_policy" "vpc-execution" {
  arn = "arn:aws:iam::aws:policy/service-role/AWSLambdaVPCAccessExecutionRole"
}

data "aws_iam_policy_document" "stream-ingest" {
  statement {
    actions = [
      "dynamodb:BatchWriteItem",
    ]

    resources = [
      "${aws_dynamodb_table.history.arn}",
    ]
  }
}

resource "aws_lambda_function" "stream-ingest" {
  function_name = "${local.prefix}-stream-ingest"
  description   = "read history audit events from kinesis stream and put it into dynamoDB"
  handler       = "lambda-stream-ingest"
  role          = "${aws_iam_role.stream-ingest.arn}"
  filename      = "${module.dummy_go_lambda.filename}"
  runtime       = "go1.x"
  timeout       = 300

  environment {
    variables = {
      HISTORY_ENVIRONMENT = "${var.environment}"
      DYNAMODB_TABLE_NAME = "${aws_dynamodb_table.history.name}"
      EVENT_SOURCE        = "history-service"
    }
  }

  vpc_config {
    subnet_ids         = "${var.twitch_subnet}"
    security_group_ids = "${var.security_group}"
  }

  lifecycle {
    ignore_changes = [
      "filename",
      "s3_key",
    ]
  }
}

data "aws_iam_policy_document" "stream-ingest-arp" {
  statement {
    actions = [
      "sts:AssumeRole",
    ]

    principals {
      type = "Service"

      identifiers = [
        "lambda.amazonaws.com",
      ]
    }
  }
}

resource "aws_iam_role" "stream-ingest" {
  name               = "${local.prefix}-stream-ingest"
  assume_role_policy = "${data.aws_iam_policy_document.stream-ingest-arp.json}"
}

resource "aws_iam_role_policy" "dynamodb-write" {
  role   = "${aws_iam_role.stream-ingest.id}"
  policy = "${data.aws_iam_policy_document.stream-ingest.json}"
}

resource "aws_iam_role_policy_attachment" "kinesis-execution" {
  role       = "${aws_iam_role.stream-ingest.id}"
  policy_arn = "${data.aws_iam_policy.kinesis-execution.arn}"
}

resource "aws_iam_role_policy_attachment" "vpc-execution" {
  role       = "${aws_iam_role.stream-ingest.id}"
  policy_arn = "${data.aws_iam_policy.vpc-execution.arn}"
}
