// Backbone of our worker for polling & processing messages

package backend

import (
	"encoding/json"
	"strings"

	"time"

	"code.justin.tv/foundation/history-worker/db"
	"code.justin.tv/foundation/history-worker/models"
	"code.justin.tv/foundation/history-worker/queue"
	"code.justin.tv/foundation/history-worker/report"
	"code.justin.tv/foundation/history-worker/utils"
)

const (
	messageReceived        = "message_received"
	messageProcessed       = "message_processed"
	parseError             = "parsing_error"
	validationError        = "validation_error"
	dynamoInsert           = "database_insert_time"
	dynamoError            = "database_insert_error"
	totalLatency           = "total_latency"
	conditionalCheckFailed = "ConditionalCheckFailedException"
	existingAuditError     = "There is already an existing audit"
	maxRetries             = 5
)

// Initial function to start the worker to poll & process received messages using callback
func StartWorker() {
	queue.Poll(processMessage)
}

// This function receives all the messages one by one & processes them.
// Returns true if message is successfully processed, or should not be retried.
func processMessage(messageBody string) bool {
	var err, db_err error

	report.StatsdThroughput(messageReceived)
	audit := &models.Audit{}

	err = json.Unmarshal([]byte(messageBody), audit)
	if err != nil {
		reportError(messageBody, parseError, err)
		return true
	}

	err = audit.Validate()
	if err != nil {
		reportError(audit.Body(), validationError, err)
		return true
	}

	db_start := time.Now()
	db_err = withRetry(audit, db.Add)

	if db_err == nil {
		report.StatsdPerformance(dynamoInsert, time.Since(db_start))
	} else {
		reportError(audit.Body(), dynamoError, db_err)
		return false
	}

	// If either DynamoDB or ES fails, we'll retry processing this
	// message by marking it as not complete
	if db_err != nil {
		return false
	}

	report.StatsdPerformance(totalLatency, time.Since(audit.CreatedAtTime))
	report.StatsdThroughput(messageProcessed)

	return true
}

// Reports error + audit information to Rollbar and Statsd
func reportError(audit_string, metric_type string, err error) {
	field := report.NewField("message", audit_string)
	report.RollbarError(err, field)
	report.StatsdThroughput(metric_type)
}

// Returns the error if the error is retriable, nil otherwise
func retriableError(err error) error {
	if err == nil {
		return nil
	}

	if strings.HasPrefix(err.Error(), conditionalCheckFailed) {
		return nil
	}

	if strings.HasPrefix(err.Error(), existingAuditError) {
		return nil
	}

	return err
}

// Calls the function passed with audit as parameter and retries on error maxRetries times
// Returns any final error returned by the input function
func withRetry(audit *models.Audit, f func(*models.Audit) error) (err error) {
	for try := 1; try <= maxRetries; try++ {
		err = retriableError(f(audit))

		if err == nil {
			break
		}

		if try < maxRetries {
			utils.BackOff(try)
		}
	}

	return err
}
