// Custom error implementation to make our errors/alerts being posted to Rollbar more generic.

package report

import (
	"github.com/stvp/rollbar"
)

// Custom error struct to hold error message, stack trace, and error level
type CustomAlert struct {
	message string
	level   string
	trace   rollbar.Stack
}

// Implementation of error interface
func (e *CustomAlert) Error() string {
	return e.message
}

// Handy function for alert type: Error
func NewErrorAlert(message string) *CustomAlert {
	return NewCustomAlert(rollbar.ERR, message)
}

// Handy function for alert type: Warning
func NewWarningAlert(message string) *CustomAlert {
	return NewCustomAlert(rollbar.WARN, message)
}

// Handy function for alert type: Debug
func NewDebugAlert(message string) *CustomAlert {
	return NewCustomAlert(rollbar.DEBUG, message)
}

// Handy function for alert type: Info
func NewInfoAlert(message string) *CustomAlert {
	return NewCustomAlert(rollbar.INFO, message)
}

// Handy function for alert type: Critical
func NewCriticalAlert(message string) *CustomAlert {
	return NewCustomAlert(rollbar.CRIT, message)
}

// Main function to instantiate custom alert struct
func NewCustomAlert(level string, message string) *CustomAlert {
	trace := rollbar.BuildStack(3) // 3 = skip 3 useless traces from top which build this alert

	return &CustomAlert{
		level:   level,
		message: message,
		trace:   trace,
	}
}

// Returns rollbar field to store additional details to send to rollbar
func NewField(name string, data interface{}) *rollbar.Field {
	return &rollbar.Field{
		Name: name,
		Data: data,
	}
}
