package main

import (
	"flag"
	"fmt"
	"math/rand"
	"net/http"
	"strconv"
	"time"

	vegeta "github.com/tsenart/vegeta/lib"
)

type RpsStep struct {
	rps      uint64
	duration time.Duration
}

type RpsPlan struct {
	startRps      uint64
	maxRps        uint64
	rampTimeRatio float64
	duration      time.Duration
	stepDuration  time.Duration
}

type UrlReader struct {
	current     string
	repeats     int
	repeatsLeft int
}

func main() {

	apiBase := flag.String("api-base", "https://api.us-east-1.twitch.tv", "Base endpoint for API")
	apiPath := flag.String("api-path", "/kraken/streams/featured", "API endpoint to hit")
	clientID := flag.String("client-id", "zoyuadrkem29ake6z6lvvh7jokg90e", "Client-ID to use for the request")

	startRps := flag.Uint64("start-rps", 10, "RPS to start load testing from")
	maxRps := flag.Uint64("max-rps", 1000, "RPS to finish load testing at")
	duration_seconds := flag.Uint64("duration", 300, "Duration in seconds for the total test")

	flag.Parse()

	duration := time.Duration(int64(*duration_seconds)) * time.Second

	headers := http.Header{}
	headers.Set("Client-Id", *clientID)

	rampTimeRatio := 1.0
	stepDuration := 30 * time.Second
	if stepDuration > duration {
		stepDuration = duration
	}

	plan := RpsPlan{startRps: *startRps,
		maxRps:        *maxRps,
		rampTimeRatio: rampTimeRatio,
		duration:      duration,
		stepDuration:  stepDuration,
	}
	steps := plan.getRpsSteps() //startRps, maxRps, duration, rampTimeRatio)
	fmt.Println(steps)

	//endpoint := apiBase + "v5/"
	endpoint := *apiBase + *apiPath
	//targeter := vegeta.NewStaticTargeter(vegeta.Target{
	//	Method: "GET",
	//	URL:    endpoint,
	//	Header: headers,
	//})

	urlReader := UrlReader{repeats: 10, repeatsLeft: 0}
	targeter := vegeta.NewLazyTargeter(urlReader, nil, headers)

	//for i := 0; i < 10; i++ {
	//	b := make([]byte, 128)
	//	urlReader.Read(b)
	//	s := string(b[:100])
	//	fmt.Println(s)
	//}
	//os.Exit(0)

	attacker := vegeta.NewAttacker()
	var metrics vegeta.Metrics
	metricsPerRps := make([]vegeta.Metrics, len(steps))

	fmt.Printf("Load testing endpoint %s from %d rps to %d rps for %f second(s)...\n\n", endpoint, plan.startRps, plan.maxRps, plan.duration.Seconds())

	for i, step := range steps {
		fmt.Printf("Load testing at %d rps for %f second(s).\n", step.rps, step.duration.Seconds())
		for res := range attacker.Attack(targeter, step.rps, step.duration) {
			fmt.Printf(res.Error)
			metrics.Add(res)
			metricsPerRps[i].Add(res)
		}
		metricsPerRps[i].Close()
		fmt.Printf("Success: %f\n", metricsPerRps[i].Success)
		fmt.Printf("50th percentile at %d rps: %s\n", step.rps, metricsPerRps[i].Latencies.P50)
		fmt.Printf("95th percentile at %d rps: %s\n", step.rps, metricsPerRps[i].Latencies.P95)
		fmt.Printf("99th percentile at %d rps: %s\n\n", step.rps, metricsPerRps[i].Latencies.P99)
	}
	metrics.Close()

	fmt.Printf("Overall 99th percentile: %s\n", metrics.Latencies.P99)
}

// Create the RPS steps, from the given RpsPlan
func (p *RpsPlan) getRpsSteps() []RpsStep {
	steps := []RpsStep{}

	rampUpDuration := int(p.duration.Seconds() * p.rampTimeRatio)

	numberOfSteps := rampUpDuration / int(p.stepDuration.Seconds())
	stepVal := uint64(int(p.maxRps-p.startRps) / numberOfSteps)

	for i := 0; i < numberOfSteps; i++ {
		step := RpsStep{rps: p.startRps + (stepVal * uint64(i)), duration: p.stepDuration}
		steps = append(steps, step)
	}

	lastStepSeconds := p.duration.Seconds() - float64(rampUpDuration)
	if lastStepSeconds == 0 {
		lastStepSeconds = p.stepDuration.Seconds()
	}
	lastStep := RpsStep{rps: p.maxRps, duration: time.Duration(lastStepSeconds) * time.Second}
	steps = append(steps, lastStep)

	return steps
}

func (u UrlReader) Read(p []byte) (n int, err error) {
	if u.repeatsLeft == 0 {
		//url := []byte(generateUrl())
		u.repeatsLeft = u.repeats
		u.current = generateUrl()
	}
	u.repeatsLeft = u.repeatsLeft - 1
	url := []byte(u.current)
	n = copy(p, url)
	return
}

func generateUrl() string {
	suffix := strconv.Itoa(rand.Int())
	apiBase := "https://api.us-east-1.twitch.tv/"
	apiUrl := "/kraken/streams/featured?_cache=" + suffix + " \n"
	return "GET " + apiBase + apiUrl

}
