package twitchclient

import (
	"errors"
	"net/http"
	"testing"

	"github.com/stretchr/testify/require"
)

func TestErrorInterface(t *testing.T) {
	var e error = &Error{Message: "Kernkraft 400", StatusCode: 400}
	require.Equal(t, "400: Kernkraft 400", e.Error())
}

func TestStatusCode(t *testing.T) {
	err1 := &Error{Message: "No status"}
	require.Equal(t, 0, StatusCode(err1))

	err2 := &Error{Message: "Kernkraft 400", StatusCode: 400}
	require.Equal(t, 400, StatusCode(err2))

	err3 := errors.New("Not a twitchclient.Error")
	require.Equal(t, 0, StatusCode(err3))
}

// Test internal helper function ErrorFromFailedResponse
func TestErrorFromFailedResponse(t *testing.T) {
	t.Run("Response is JSON with a message field", func(t *testing.T) {
		resp := &http.Response{StatusCode: 401, Header: http.Header{}}
		resp.Header.Add("Content-Type", "application/json")
		resp.Body = bodyStr(`{ "Message": "Hello", "StatusCode": 500 }`)
		err := ErrorFromFailedResponse(resp)
		require.Equal(t, 401, err.StatusCode)  // StatusCode is set from response status, not from the JSON StatusCode field
		require.Equal(t, "Hello", err.Message) // message was successfully parsed
		require.Equal(t, "", err.ErrorCode)    // error_code is empty by default
	})
	t.Run("Response is JSON with also an error_code field", func(t *testing.T) {
		resp := &http.Response{StatusCode: 401, Header: http.Header{}}
		resp.Header.Add("Content-Type", "application/json")
		resp.Body = bodyStr(`{ "message": "Bye Bye", "error_code": "THOU_SHALL_NO_PASS" }`)
		err := ErrorFromFailedResponse(resp)
		require.Equal(t, 401, err.StatusCode)                 // StatusCode is set from response status
		require.Equal(t, "THOU_SHALL_NO_PASS", err.ErrorCode) // error_code was successfully parsed
		require.Equal(t, "Bye Bye", err.Message)              // message was successfully parsed
	})
	t.Run("Response content type is application/json; charset=utf8", func(t *testing.T) {
		resp := &http.Response{StatusCode: 401, Header: http.Header{}}
		resp.Header.Add("Content-Type", "application/json; charset=utf8")
		resp.Body = bodyStr(`{ "message": "Hello" }`)
		err := ErrorFromFailedResponse(resp)
		require.Equal(t, 401, err.StatusCode)
		require.Equal(t, "Hello", err.Message) // message was successfully parsed
		require.Equal(t, "", err.ErrorCode)    // error_code is empty by default
	})
	t.Run("Response is JSON but has no message field", func(t *testing.T) {
		resp := &http.Response{StatusCode: 401, Header: http.Header{}}
		resp.Header.Add("Content-Type", "application/json")
		resp.Body = bodyStr(`{ "other": "Hello", "StatusCode": 499 }`)
		err := ErrorFromFailedResponse(resp)
		require.Equal(t, 401, err.StatusCode)
		require.Equal(t, "", err.Message)   // message is empty
		require.Equal(t, "", err.ErrorCode) // error_code is empty
	})
	t.Run("Response content type is application/json but the JSON is malformed", func(t *testing.T) {
		resp := &http.Response{StatusCode: 401, Header: http.Header{}}
		resp.Header.Add("Content-Type", "application/json")
		resp.Body = bodyStr(`{ no_quotes: "is not json" }`)
		err := ErrorFromFailedResponse(resp)
		require.Equal(t, 401, err.StatusCode)
		require.Equal(t, "Unable to decode JSON response: invalid character 'n' looking for beginning of object key string", err.Message) // parsed as plain text
	})
	t.Run("Response content type is not supported", func(t *testing.T) {
		resp := &http.Response{StatusCode: 401, Header: http.Header{}}
		resp.Header.Add("Content-Type", "invalid/content/type")
		resp.Body = bodyStr(`{ "message": "Hello" }`)
		err := ErrorFromFailedResponse(resp)
		require.Equal(t, 401, err.StatusCode)
		require.Equal(t, `{ "message": "Hello" }`, err.Message) // parsed as plain text
	})
	t.Run("Response has no content type", func(t *testing.T) {
		resp := &http.Response{StatusCode: 404, Header: http.Header{}}
		resp.Body = bodyStr("anything goes")
		err := ErrorFromFailedResponse(resp)
		require.Equal(t, 404, err.StatusCode)
		require.Equal(t, "anything goes", err.Message)
	})
	t.Run("Response is unreadable plain text", func(t *testing.T) {
		resp := &http.Response{StatusCode: 200, Header: http.Header{}}
		resp.Body = &readErrorReadCloser{}
		err := ErrorFromFailedResponse(resp)
		require.Equal(t, 200, err.StatusCode)
		require.Equal(t, "Unable to read response body: unexpected EOF", err.Message)
	})
}

func TestHandleFailedResponse(t *testing.T) {
	// For backwards compatibility, it should keep working as before
	resp := &http.Response{StatusCode: 401, Header: http.Header{}}
	resp.Header.Add("Content-Type", "application/json")
	resp.Body = bodyStr(`{ "message": "Hello", "error_code": "backwards_compatible"}`)
	err := HandleFailedResponse(resp)
	require.Equal(t, 401, err.StatusCode)
	require.Equal(t, "Hello", err.Message)
	require.Equal(t, "backwards_compatible", err.ErrorCode)
}
