package twitchclient_test

import (
	"context"
	"testing"

	"code.justin.tv/foundation/twitchclient"
	"github.com/stretchr/testify/require"
)

func TestWithDefaults(t *testing.T) {
	reqOpts := twitchclient.ReqOpts{
		AuthorizationToken: "thisisatoken",
		StatName:           "mycoolstatname",
	}
	defaults := twitchclient.ReqOpts{
		StatName:       "default_stat_name",
		StatSampleRate: float32(0.6),
	}

	merged := (&reqOpts).WithDefaults(defaults)
	require.Equal(t, "thisisatoken", merged.AuthorizationToken, "AuthorizationToken has original value")
	require.Equal(t, "mycoolstatname", merged.StatName, "StatName has original value, not the default")
	require.Equal(t, float32(0.6), merged.StatSampleRate, "StatSampleRate is the default value")
	require.Equal(t, "", merged.ClientID, "ClientID has no value, because is not defined as default value either")
}

func TestCtxWithReqOpts(t *testing.T) {
	ctx := context.Background()
	ctx = twitchclient.WithReqOpts(ctx, twitchclient.ReqOpts{
		AuthorizationToken: "thisisatoken",
		StatName:           "mycoolstatname",
	})
	ctxOpts := twitchclient.GetReqOpts(ctx)
	require.Equal(t, "thisisatoken", ctxOpts.AuthorizationToken, "ctxOpts.AuthorizationToken")
	require.Equal(t, "mycoolstatname", ctxOpts.StatName, "ctxOpts.StatName")
	require.Equal(t, "", ctxOpts.ClientID, "ctxOpts.ClientID is empty")

	// Merge some more options
	ctx = twitchclient.WithReqOpts(ctx, twitchclient.ReqOpts{
		AuthorizationToken: "new-token",
		ClientID:           "1234",
	})
	ctxOpts = twitchclient.GetReqOpts(ctx)
	require.Equal(t, "new-token", ctxOpts.AuthorizationToken, "ctxOpts.AuthorizationToken has new value")
	require.Equal(t, "mycoolstatname", ctxOpts.StatName, "ctxOpts.StatName keeps previous value")
	require.Equal(t, "1234", ctxOpts.ClientID, "ctxOpts.ClientID has new value")
}

func TestWithClearReqOpts(t *testing.T) {
	ctx := context.Background()
	ctx = twitchclient.WithReqOpts(ctx, twitchclient.ReqOpts{
		AuthorizationToken: "thisisatoken",
	})
	ctxOpts := twitchclient.GetReqOpts(ctx)
	require.Equal(t, "thisisatoken", ctxOpts.AuthorizationToken, "ctxOpts.AuthorizationToken")

	ctx = twitchclient.WithClearReqOpts(ctx)
	ctxOpts = twitchclient.GetReqOpts(ctx)
	require.Equal(t, "", ctxOpts.AuthorizationToken, "ctxOpts.AuthorizationToken is clear")
}
