package mocks

import (
	"sync"
	"time"

	"github.com/cactus/go-statsd-client/statsd"
)

// NewStatter builds a fake statter that tracks usage so it can be
// asserted and used for testing.
func NewStatter() *Statter {
	statter, err := statsd.NewNoopClient()
	if err != nil {
		return nil
	}
	s := &Statter{Statter: statter}
	s.Reset()
	return s
}

// Statter implements a statsd.Statter as a noop and tracks usage for tests.
type Statter struct {
	statsd.Statter
	mu sync.Mutex

	Timings       map[string]int     // number of times TimingDuration has been called for each stat.
	TimingsSample map[string]float32 // sample used on last call to TimingDuration on a given stat.

	Incs       map[string]int64   // total incremented for each stat after calling Inc(stat, n, sample).
	IncsSample map[string]float32 // sample used on last call to Inc on a given stat.
}

var _ statsd.Statter = (*Statter)(nil) // make sure it implements the Statter interface

// Reset all counters back to zero
func (s *Statter) Reset() {
	s.Timings = map[string]int{}
	s.TimingsSample = map[string]float32{}
	s.Incs = map[string]int64{}
	s.IncsSample = map[string]float32{}
}

// TimingDuration tracks Timings and TimingsSample
func (s *Statter) TimingDuration(stat string, d time.Duration, sample float32) error {
	s.mu.Lock()
	defer s.mu.Unlock()

	s.Timings[stat]++
	s.TimingsSample[stat] = sample
	return nil
}

// TimingCounts returns the number of times that TimingDuration was called with that stat
func (s *Statter) TimingCounts(stat string) int {
	return s.Timings[stat]
}

// AnyTimingCounts returns the number ot times that TimingDuration on any stat
func (s *Statter) AnyTimingCounts() int {
	sum := 0
	for _, count := range s.Timings {
		sum += count
	}
	return sum
}

// TimingSample returns the sample rate used on the last call to TimingCounts(stat)
func (s *Statter) TimingSample(stat string) float32 {
	return s.TimingsSample[stat]
}

// Inc tracks Incs and IncsSample
func (s *Statter) Inc(stat string, count int64, sample float32) error {
	s.mu.Lock()
	defer s.mu.Unlock()

	s.Incs[stat] += count
	s.IncsSample[stat] = sample
	return nil
}

// IncSum returns the total count incremented by calling Inc(stat)
func (s *Statter) IncSum(stat string) int64 {
	return s.Incs[stat]
}

// IncSample returns the sample rate used on the last call to Inc(stat)
func (s *Statter) IncSample(stat string) float32 {
	return s.IncsSample[stat]
}
