package twitchserver_test

import (
	"context"
	"fmt"
	"log"
	"net/http"
	"time"

	"code.justin.tv/foundation/twitchserver"

	"goji.io"
	"goji.io/pat"
)

type MyServer struct {
	*goji.Mux
	logger *requestIDLogger
}

func ExampleListenAndServe() {
	server := twitchserver.NewServer()
	dimensionKey := new(int)
	elevateKey := new(int)
	logger := &requestIDLogger{dimensionKey, elevateKey}
	logger.Log("Created a logger")
	s := &MyServer{server, logger}
	s.HandleFunc(pat.Get("/foo"), s.endpoint)

	// this will add default handlers for OS signals
	twitchserver.AddDefaultSignalHandlers()
	config := &twitchserver.ServerConfig{
		DimensionKey: dimensionKey,
		ElevateKey:   elevateKey,
	}

	go func() {
		err := twitchserver.ListenAndServe(s, config)
		if err != nil {
			log.Fatal(err)
		}
	}()

	time.Sleep(200 * time.Millisecond)
	headers := http.Header{}
	// x-amzn-trace-id is automatically added by twitchclient
	headers.Add("x-amzn-trace-id", "Root=normal-request")
	doReq(headers)

	headers = http.Header{}
	headers.Add("X-Ctxlog-Elevate", "true")
	headers.Add("x-amzn-trace-id", "Root=elevated-request")
	doReq(headers)

	// Output:
	// [LOG]: Created a logger
	// [LOG]: normal-request The server is handling a request.
	// [DEBUG]: normal-request This is a debug line
	// [LOG]: elevated-request The server is handling a request.
	// [LOG]: elevated-request This is a debug line
}

func doReq(h http.Header) {
	req, err := http.NewRequest(http.MethodGet, "http://localhost:8000/foo", nil)
	if err != nil {
		log.Fatal(err)
	}
	req.Header = h
	client := http.Client{}
	_, err = client.Do(req)
	if err != nil {
		log.Fatal(err)
	}
}

func (s *MyServer) endpoint(w http.ResponseWriter, r *http.Request) {
	s.logger.LogCtx(r.Context(), "The server is handling a request.")
	s.logger.DebugCtx(r.Context(), "This is a debug line")
	_, err := w.Write([]byte("OK"))
	if err != nil {
		log.Print("Failed to write a response.")
	}
}

// requestIDLogger is an implementation of a logger which will prepend a request ID
// to each log line in the simplest way possible. These examples exist mostly to show
// how you can adapt an existing logging implementation to the twitchserver / client interfaces.
// You may want to consider using a logger which already implements context logging functions,
// like ElevatedLog from https://git-aws.internal.justin.tv/feeds/log.
type requestIDLogger struct {
	DimensionKey interface{}
	ElevateKey   interface{}
}

func (l *requestIDLogger) LogCtx(ctx context.Context, params ...interface{}) {
	requestID := l.requestID(ctx)
	fmt.Print("[LOG]:", fmt.Sprintf(" %s ", requestID), fmt.Sprintln(params...))
}

// Some logs occur outside of a request path
func (l *requestIDLogger) Log(params ...interface{}) {
	fmt.Print("[LOG]: ", fmt.Sprintln(params...))
}

// You may want to elevate logs for some requests. For example, you could sample a small amount
// of your logs with verbose logging.
func (l *requestIDLogger) DebugCtx(ctx context.Context, params ...interface{}) {
	if l.isElevated(ctx) {
		l.LogCtx(ctx, params...)
		return
	}
	requestID := l.requestID(ctx)
	fmt.Print("[DEBUG]:", fmt.Sprintf(" %s ", requestID), fmt.Sprintln(params...))
}

func (l *requestIDLogger) isElevated(ctx context.Context) bool {
	val, ok := ctx.Value(l.ElevateKey).(bool)
	return ok && val
}

// requestID extracts a request ID from the provided context. It returns an empty
// string if it was not found. You might want to consider just logging everything
// in the dimension key.
func (l *requestIDLogger) requestID(ctx context.Context) string {
	val := ctx.Value(l.DimensionKey)
	if val == nil {
		return ""
	}
	dimensions, ok := val.([]interface{})
	if !ok {
		return ""
	}

	for i := 0; i < len(dimensions); i += 2 {
		if i == len(dimensions)-1 {
			continue
		}
		key, isString := dimensions[i].(string)
		if !isString {
			continue
		}
		if key == "X-Ctxlog-LogID" {
			logID, isString := dimensions[i+1].(string)
			if !isString {
				return ""
			}
			return logID
		}
	}
	return ""
}
