package api

import (
	"encoding/json"
	"errors"
	"net/http"

	"code.justin.tv/video/spectre/backend"

	"github.com/zenazn/goji/web"
)

var ErrChannelIsEnabled = errors.New("api: cannot modify a channel that is already enabled")

func (T *Router) ShowPlaylist(c web.C, w http.ResponseWriter, r *http.Request) {
	channelID, err := idToInt(c.URLParams["channel_id"])
	if err != nil {
		T.ServeError(w, r, err)
		return
	}
	playlistID, err := idToInt(c.URLParams["playlist_id"])
	if err != nil {
		T.ServeError(w, r, err)
		return
	}
	playlist, err := T.Backend.GetPlaylist(channelID, playlistID)
	if err != nil {
		T.ServeError(w, r, err)
		return
	}
	err = json.NewEncoder(w).Encode(playlist)
	if err != nil {
		T.ServeError(w, r, err)
	}
}

func (T *Router) CreatePlaylist(c web.C, w http.ResponseWriter, r *http.Request) {
	channelID, err := idToInt(c.URLParams["channel_id"])
	if err != nil {
		T.ServeError(w, r, err)
		return
	}

	err = T.AuthHandler.AuthorizeRequest(r.Header.Get("Authorization"), channelID, r, T.StatsdClient)
	if err != nil {
		T.ServeError(w, r, err)
		return
	}

	params := backend.PlaylistParams{}
	err = json.NewDecoder(r.Body).Decode(&params)
	if err != nil {
		T.ServeError(w, r, err)
		return
	}
	params.ChannelID = channelID

	playlist, err := T.Backend.CreatePlaylist(params)

	if err != nil {
		T.ServeError(w, r, err)
		return
	}
	err = json.NewEncoder(w).Encode(playlist)
	if err != nil {
		T.ServeError(w, r, err)
	}
}

func (T *Router) UpdatePlaylist(c web.C, w http.ResponseWriter, r *http.Request) {
	channelID, err := idToInt(c.URLParams["channel_id"])
	if err != nil {
		T.ServeError(w, r, err)
		return
	}
	playlistID, err := idToInt(c.URLParams["playlist_id"])
	if err != nil {
		T.ServeError(w, r, err)
		return
	}

	err = T.AuthHandler.AuthorizeRequest(r.Header.Get("Authorization"), channelID, r, T.StatsdClient)
	if err != nil {
		T.ServeError(w, r, err)
		return
	}

	channel, err := T.Backend.GetChannelUncached(channelID)
	if err != nil {
		T.ServeError(w, r, err)
		return
	}
	if channel.IsEnabled() {
		T.ServeError(w, r, ErrChannelIsEnabled)
		return
	}

	params := backend.PlaylistParams{}
	err = json.NewDecoder(r.Body).Decode(&params)
	if err != nil {
		T.ServeError(w, r, err)
		return
	}
	params.ID = playlistID
	params.ChannelID = channelID

	playlist, err := T.Backend.UpdatePlaylist(params)

	if err != nil {
		T.ServeError(w, r, err)
		return
	}
	w.WriteHeader(http.StatusOK)
	err = json.NewEncoder(w).Encode(playlist)
	if err != nil {
		T.ServeError(w, r, err)
	}

}

func (T *Router) DestroyPlaylist(c web.C, w http.ResponseWriter, r *http.Request) {
	channelID, err := idToInt(c.URLParams["channel_id"])
	if err != nil {
		T.ServeError(w, r, err)
		return
	}
	playlistID, err := idToInt(c.URLParams["playlist_id"])
	if err != nil {
		T.ServeError(w, r, err)
		return
	}

	err = T.AuthHandler.AuthorizeRequest(r.Header.Get("Authorization"), channelID, r, T.StatsdClient)
	if err != nil {
		T.ServeError(w, r, err)
		return
	}

	channel, err := T.Backend.GetChannelUncached(channelID)
	if err != nil {
		T.ServeError(w, r, err)
		return
	}
	if channel.IsEnabled() {
		T.ServeError(w, r, ErrChannelIsEnabled)
		return
	}

	err = T.Backend.DeletePlaylist(channelID, playlistID)
	if err != nil {
		T.ServeError(w, r, err)
		return
	}
}
