package authtest

import (
	"net/http"
	"testing"
	"time"

	. "github.com/smartystreets/goconvey/convey"

	"code.justin.tv/common/goauthorization"
	"code.justin.tv/common/golibs/statsd"
	"code.justin.tv/video/spectre/auth"
)

func performAuthorizeRequest(headers map[string]string, currentChannel int) error {
	goauthorization.InitializeDecoder("spectre.pub", "code.justin.tv/video/spectre", "code.justin.tv/video/spectre")
	authHandler := auth.NewHandler("../rails_key.pub")

	req := &http.Request{}
	req.Header = http.Header{}
	for header, token := range headers {
		req.Header.Set(header, token)
	}

	return authHandler.AuthorizeRequest(headers["Authorization"], currentChannel, req, statsd.Noop())
}

func TestAuthorizeRequest(t *testing.T) {
	goauthorization.InitializeEncoder("spectre.pem", "code.justin.tv/video/spectre")

	currentChannel := 50295775

	authorizationClaims := goauthorization.CapabilityClaims{
		"edit_channel": goauthorization.CapabilityClaim{
			"channel_id": currentChannel,
		},
	}
	cartmanToken := goauthorization.GenerateToken(
		time.Date(2025, time.January, 1, 1, 0, 0, 0, time.UTC),
		time.Date(2015, time.January, 1, 1, 0, 0, 0, time.UTC),
		[]string{"code.justin.tv/video/spectre"},
		string(currentChannel),
		authorizationClaims,
	)

	Convey("returns no error for a valid Cartman token", t, func() {
		headers := map[string]string{
			"Twitch-Authorization": cartmanToken.String(),
		}
		err := performAuthorizeRequest(headers, currentChannel)
		So(err, ShouldBeNil)
	})

	Convey("returns ErrAuthorizationTokenRequired if an invalid Cartman token is provided", t, func() {
		headers := map[string]string{
			"Twitch-Authorization": "cartmanToken",
		}
		err := performAuthorizeRequest(headers, currentChannel)
		So(err, ShouldEqual, auth.ErrAuthorizationTokenRequired)
	})

	Convey("returns no error for an invalid Cartman token and valid legacy token", t, func() {
		headers := map[string]string{
			"Twitch-Authorization": "cartmanToken",
			"Authorization":        "eyJ0eXAiOiJKV1QiLCJhbGciOiJSUzI1NiJ9.eyJkYXRhIjp7Im9iamVjdHMiOls1MDI5NTc3NV0sImFjdGlvbnMiOlsiZWRpdCJdLCJ2ZXJzaW9uIjoiMC4xIn0sImV4cCI6MTczNTY4OTYwMCwic3ViIjo1MDI5NTc3NSwiaXNzIjoiY29kZS5qdXN0aW4udHYvd2ViL2NhcnRtYW4iLCJhdWQiOlsiY29kZS5qdXN0aW4udHYvdmlkZW8vc3BlY3RyZSJdLCJuYmYiOjE0NDU2MjY4MDZ9.dXRVAwUnsidq1pjxijjGCX0kGikXO91x47VV6Z0802cLMRRi8-fIQpESAJ5YDA8XbHKT3mPQqhDPbX066qEE7glo_gTRHTGiMBzB_bBbpU1Dm6W0sREZC7VXws-CNjCXOI0h4Ssgfa9raJ07jxzolPXpuMek8fkclyNIiOxQ2ClfW3wu3V9bfZXFQn29G5A9YXJEbtV0kFYfR1-RRRfNkbxW59Ovp-rrkDtx0avba2W5m4JO8N8FuZ1YSmve0VZr5-JOvr8XSiCQUj77WUT7buXjv06FSwcYppB3b1Ni7jR6ZxEXmWimUc_PYU_SCpxZescPLTzX3tHwCX5lXsK6VxGLH7CHJY_n0jk6XgANAwWanMqf1hDhu683BbQo0JfBmHb75PXkAOp1l_VybQkCsiTkF4_ku-3UfvHiFHvZ-sl-KI13zgcLIRjYtJ_ZkKPmwgA6cYaPS1WXP1-ZA-LPq9_tyD7LrdGBeop-x0VUWuS83AyhCCjQ42iYdlMrsWogu9FFpipx6Wqv-y40DQwUMc98Wl3cWxEQdpY68ggMlvutzd5cVfBBiFVIYiLqv5Q3vOiuQFARyZFkLa0mhpt9s3_4cGXf_to0FmiRnmFKnLHbNEr88X1J0ximpmL0YK7J5RWLkzdn5rp1jAJpYYvENiNyPU9rqRyzovJOiGdNgEQ",
		}
		err := performAuthorizeRequest(headers, currentChannel)
		So(err, ShouldBeNil)
	})

}
