module "base" {
    source = "git::git+ssh://git@git-aws.internal.justin.tv/release/terraform.git//base"
}

# Should be moved to a base module
provider "aws" {
  region = "us-west-2"
}

provider "consul" {
    address = "consul.internal.justin.tv"
    datacenter = "us-west2"
}

variable env {
  default = "development"
}

variable owner {
  default = "golf"
}

variable count {
  default = "1"
}

variable dev_avail {
    default {
        zone0 = "subnet-d4a423b1"
        zone1 = "subnet-67ad0810"
        zone2 = "subnet-894ab1d0"
        # Hack to make this work with larger counts
        zone3  = "subnet-d4a423b1"
        zone4  = "subnet-67ad0810"
        zone5  = "subnet-894ab1d0"
        zone6  = "subnet-d4a423b1"
        zone7  = "subnet-67ad0810"
        zone8  = "subnet-894ab1d0"
        zone9  = "subnet-d4a423b1"
        zone10 = "subnet-67ad0810"
        zone11 = "subnet-894ab1d0"
        zone12 = "subnet-d4a423b1"
        zone13 = "subnet-67ad0810"
        zone14 = "subnet-894ab1d0"
        zone15 = "subnet-d4a423b1"
        zone16 = "subnet-67ad0810"
        zone17 = "subnet-894ab1d0"
        zone18 = "subnet-d4a423b1"
        zone19 = "subnet-67ad0810"
        zone20 = "subnet-894ab1d0"
    }
}

################
# Spectre Server #
################

module "spectre-instance" {
    source = "git::git+ssh://git@git-aws.internal.justin.tv/release/terraform.git//puppet_instance"
    hostname_prefix = "${var.env}-spectre"
    cluster = "spectre"
    owner = "${var.owner}"
    facts = "app_env=${var.env}"
}

resource "aws_route53_record" "spectre" {
  count = 1
  zone_id = "${module.base.r53}"
  name = "spectre-${count.index}.prod.us-west2.justin.tv"
  type = "A"
  ttl = "300"
  records = ["${element(aws_instance.spectre.*.private_ip, count.index)}"]
}

resource "aws_instance" "spectre" {
  count = "1"
  ami = "ami-2f82b11f"
  instance_type = "t2.medium"
  subnet_id = "subnet-faa4239f"
  # subnet_id = "${lookup(var.avail, concat("zone", count.index))}"
  security_groups = ["${module.base.prod_twitch_security_group}"]
  user_data = <<END_OF_STRING
#cloud-config
hostname: "spectre-${count.index}"
fqdn: "spectre-${count.index}.prod.us-west2.justin.tv"
manage_etc_hosts: true
runcmd:
 - sleep 1
 - echo "cluster=spectre" > /etc/facter/facts.d/cluster.txt
 - echo "twitch_environment=production" > /etc/facter/facts.d/twitch_environment.txt
 - echo "clean=true" > /etc/facter/facts.d/clean.txt
 - echo "app_env=spectre" > /etc/facter/facts.d/app_env.txt
 - puppet agent --test
END_OF_STRING
    tags {
        Name = "spectre-${count.index}.prod.us-west2.justin.tv"
        Owner = "devtools"
        Email = "devtools@twitch.tv"
    }
}

######
# DB #
######

resource "aws_db_subnet_group" "spectre" {
	name = "spectre-${var.env}"
	description = "DB subnet for spectre; env: '${var.env}'"

	subnet_ids = ["${var.dev_avail.zone0}", "${var.dev_avail.zone1}", "${var.dev_avail.zone2}"]
}

resource "aws_db_instance" "spectre" {
	identifier = "spectre-${var.env}"
	allocated_storage = 5
	engine = "postgres"
	engine_version = "9.3.5"
	instance_class = "db.t2.medium"
	multi_az = true
	name = "spectre"
	port = 5432
	username = "spectre"
	password = "vwiehrpiacnldmn"
	vpc_security_group_ids = ["${module.base.dev_twitch_security_group}"]
	db_subnet_group_name = "${aws_db_subnet_group.spectre.name}"
	parameter_group_name = "default.postgres9.3"
}

##################################
# Register environment in consul #
##################################

resource "consul_keys" "spectre-env" {
    datacenter = "us-west2"
    key {
        name = "env-registration"
        path = "dynamic-envs/video/spectre/${var.env.name}"
        value = "{}"
        delete = true
    }
}

###############################
# Configure the App in Consul #
###############################

resource "consul_keys" "spectre-db-password" {
    datacenter = "us-west2"
    key {
        name = "spectre-db-password"
        path = "settings/spectre/${var.env.name}/pg-conn-info"
        value = "host=${aws_db_instance.spectre.address} port=5432 user=spectre password=vwiehrpiacnldmn"
        delete = true
    }
}

##########
# Output #
##########

output "url" {
  value = "http://${module.spectre-instance.fqdn}:8080/"
}
