package main

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"os"
	"strconv"
	"strings"
	"time"
)

const (
	batchSize           = 500
	clueGetHostEndpoint = "http://clue-internal.production.us-west2.twitch.tv:6100/hosts?host=%s"
	jaxEndpoint         = "http://jax-internal-production.us-west2.justin.tv/streams?size=%d&from=%d&fields=rails.channel_id"
)

type Hit struct {
	Channel    string                    `json:"channel"`
	Properties map[string]map[string]int `json:"properties"`
}

type JaxResponse struct {
	Total int   `json:"_total"`
	Hits  []Hit `json:"hits"`
}

type Host struct {
	HostId   int `json:"host_id"`
	TargetId int `json:"target_id"`
}

type ClueResponse struct {
	Hosts []Host `json:"hosts"`
}

func main() {
	// get total number of channels from jax
	url := fmt.Sprintf(jaxEndpoint, 20, 0)
	fmt.Printf("calling jax at %v\n", url)
	res, err := http.Get(url)
	if err != nil {
		log.Fatal(err)
	}

	data, err := ioutil.ReadAll(res.Body)
	if err != nil {
		log.Fatal(err)
	}
	res.Body.Close()
	var response JaxResponse
	err = json.Unmarshal(data, &response)
	if err != nil {
		log.Fatal(err)
	}

	f, err := os.Create("../live_hosting.log")
	if err != nil {
		log.Fatal(err)
	}

	defer f.Close()

	for i := 0; i < response.Total; i += batchSize {
		// get batches of live channels from jax
		jaxUrl := fmt.Sprintf(jaxEndpoint, batchSize, i)
		fmt.Printf("calling jax at %v\n", jaxUrl)
		res, err := http.Get(jaxUrl)
		if err != nil {
			log.Fatal(err)
		}

		data, err := ioutil.ReadAll(res.Body)
		if err != nil {
			log.Fatal(err)
		}
		res.Body.Close()
		var response JaxResponse
		err = json.Unmarshal(data, &response)
		if err != nil {
			log.Fatal(err)
		}

		channelIds := make([]string, 0, len(response.Hits))
		for _, hit := range response.Hits {
			railsProperty, ok := hit.Properties["rails"]
			if !ok {
				continue
			}
			channelId, ok := railsProperty["channel_id"]
			if !ok || channelId == 0 {
				continue
			}

			channelIds = append(channelIds, strconv.Itoa(channelId))
		}

		// get live channels who are currently hosting
		channelIdsStr := strings.Join(channelIds[:], ",")
		clueGetHostUrl := fmt.Sprintf(clueGetHostEndpoint, channelIdsStr)
		clueRes, err := http.Get(clueGetHostUrl)
		if err != nil {
			log.Fatal(err)
		}

		clueData, err := ioutil.ReadAll(clueRes.Body)
		if err != nil {
			log.Fatal(err)
		}
		clueRes.Body.Close()
		var clueResponse ClueResponse
		err = json.Unmarshal(clueData, &clueResponse)
		if err != nil {
			log.Fatal(err)
		}

		for _, host := range clueResponse.Hosts {
			if host.TargetId == 0 {
				continue
			}
			log.Printf("found host - host_id = %v, target_id = %v\n", host.HostId, host.TargetId)
			_, err := f.WriteString(fmt.Sprintf("%v\n", host.HostId))
			if err != nil {
				log.Print(err)
			}
		}
		time.Sleep(10 * time.Millisecond)
	}
}
